/**
 * @file pc_conversions.h
 * @author Jake Greaves
 * @date 15 Dec 2017
 * @brief Conversions between config file values and register settings
 */ 


#ifndef __PC_CONVERSIONS_H_
#define __PC_CONVERSIONS_H_


#include "adi_sense_config_types.h"


/*! Calculate sizeof struct array containing possible values */
#if !defined(ARRAY_SIZE)
#define ARRAY_SIZE(x) (sizeof((x)) / sizeof((x)[0]))
#endif


/*! Struct typedef of possible config values, coupled with equivalent register values */
typedef struct {
    	char *stringVal;
    	int32_t structVal;
} StringToStructVals;

/*! Struct containing config file keyword, paired with the conversion struct array and the array size */
struct keywordConvert {
	char *stringEquiv;
	StringToStructVals *valsConvert;
	unsigned int valCount;
};


/*! ProductID Values */
static StringToStructVals productIdVals[] = {
	{"N/A", 				ADI_SENSE_PRODUCT_ID_1000},
	{"0x0020", 				ADI_SENSE_PRODUCT_ID_1000}
};

/*! OperationalMode Values */
static StringToStructVals operationalModeVals[] = {
	{"N/A",        			ADI_SENSE_1000_OPERATING_MODE_SINGLECYCLE},
	{"SingleCycle",        		ADI_SENSE_1000_OPERATING_MODE_SINGLECYCLE},
	{"ContinuousConversion",  	ADI_SENSE_1000_OPERATING_MODE_CONTINUOUS},
	{"MultiCycle",            	ADI_SENSE_1000_OPERATING_MODE_MULTICYCLE}
};

/*! PowerMode Values */
static StringToStructVals powerModeVals[] = {
	{"N/A",     			ADI_SENSE_1000_POWER_MODE_LOW},
	{"Low",     			ADI_SENSE_1000_POWER_MODE_LOW},
	{"Mid",     			ADI_SENSE_1000_POWER_MODE_MID},
	{"Full",    			ADI_SENSE_1000_POWER_MODE_FULL}
};

/*! CycleTime Values */
static StringToStructVals cycleTimeVals[] = {
	{"N/A",     			0}
};

/*! DataReadyMode Values */
static StringToStructVals dataReadyModeVals[] = {
	{"N/A", 				ADI_SENSE_1000_DATAREADY_PER_CONVERSION},
	{"PerMeasurement", 		ADI_SENSE_1000_DATAREADY_PER_CONVERSION},
	{"PerCycle", 			ADI_SENSE_1000_DATAREADY_PER_CYCLE},
	{"PerFifoFill", 			ADI_SENSE_1000_DATAREADY_PER_MULTICYCLE_BURST}
};

/*! GlobalDiagnostic Values */
static StringToStructVals globalDiagnosticVals[] = {
	{"N/A", 				1},
	{"False", 				1},
	{"True", 				0}
};

/*! MeasurementDiagnostic Values */
static StringToStructVals measurementDiagnosticVals[] = {
	{"N/A", 				1},
	{"False", 				1},
	{"True", 				0}
};

/*! DiagnosticFrequency Values */
static StringToStructVals diagnosticFrequencyVals[] = {
	{"N/A", 				ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS_DISABLED},
	{"None", 				ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS_DISABLED},
	{"PerCycle", 			ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS_PER_CYCLE},
	{"Per100Cycles", 			ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS_PER_100_CYCLES},
	{"Per1000Cycles",			ADI_SENSE_1000_OPEN_SENSOR_DIAGNOSTICS_PER_1000_CYCLES}
};

/*! EnableChannel Values */
static StringToStructVals enableChannelVals[] = {
	{"N/A", 				0},
	{"False", 				0},
	{"True",  				1}
};

/*! MeasurementsPerCycle Values */
static StringToStructVals measurementsPerCycleVals[] = {
	{"N/A", 				1}
};

/*! MeasurementsPerCycle Values */
static StringToStructVals settlingTimeVals[] = {
	{"N/A", 				0}
};

/*! PublishMeasurement Values */
static StringToStructVals publishMeasurementVals[] = {
	{"N/A", 				0},
	{"False", 				1},
	{"True",  				0}
};
/*! ChannelID Values */
static StringToStructVals channelIdVals[] = {
	{"N/A",             		ADI_SENSE_1000_CHANNEL_ID_NONE},
	{"Cjc0",            		ADI_SENSE_1000_CHANNEL_ID_CJC_0},
	{"Cjc1",            		ADI_SENSE_1000_CHANNEL_ID_CJC_1},
	{"Sensor0",        		ADI_SENSE_1000_CHANNEL_ID_SENSOR_0},
	{"Sensor1",        		ADI_SENSE_1000_CHANNEL_ID_SENSOR_1},
	{"Sensor2",        		ADI_SENSE_1000_CHANNEL_ID_SENSOR_2},
	{"Sensor3",        		ADI_SENSE_1000_CHANNEL_ID_SENSOR_3},
	{"Voltage0",       		ADI_SENSE_1000_CHANNEL_ID_VOLTAGE_0},
	{"Current0",        		ADI_SENSE_1000_CHANNEL_ID_CURRENT_0},
	{"I2c0",           		ADI_SENSE_1000_CHANNEL_ID_I2C_0},
	{"I2c1",           		ADI_SENSE_1000_CHANNEL_ID_I2C_1},
	{"Spi0",           		ADI_SENSE_1000_CHANNEL_ID_SPI_0},
	{"Spi1",           		ADI_SENSE_1000_CHANNEL_ID_SPI_1},
	{"Spi2",           		ADI_SENSE_1000_CHANNEL_ID_SPI_2}
};

/*! UnitTranslation Values */
static StringToStructVals unitTranslationVals[] = {
	{"N/A",         			ADI_SENSE_1000_MEASUREMENT_UNIT_DEFAULT},
	{"Celsius",         		ADI_SENSE_1000_MEASUREMENT_UNIT_CELSIUS},
	{"Fahrenheit",      		ADI_SENSE_1000_MEASUREMENT_UNIT_FAHRENHEIT},
	{"Psi",         			ADI_SENSE_1000_MEASUREMENT_UNIT_DEFAULT}, //handle as default
	{"%",      				ADI_SENSE_1000_MEASUREMENT_UNIT_DEFAULT}	//handle as default
};

/*! SensorID Values */
static StringToStructVals sensorVals[] = {
	{"N/A",  				ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_T_DEF_L1},
	{"Thermocouple.T.Def.L1",  	ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_T_DEF_L1},
	{"Thermocouple.J.Def.L1",  	ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_J_DEF_L1},
	{"Thermocouple.K.Def.L1",  	ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_K_DEF_L1},
	
	{"RTD.2W.PT100.Def.L1",   	ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_PT100_DEF_L1},
	{"RTD.2W.PT1000.Def.L1",  	ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_PT1000_DEF_L1},
	{"RTD.3W.PT100.Def.L1", 	ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_PT100_DEF_L1},
	{"RTD.3W.PT1000.Def.L1", 	ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_PT1000_DEF_L1},
	{"RTD.4W.PT100.Def.L1", 	ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_PT100_DEF_L1},
	{"RTD.4W.PT1000.Def.L1", 	ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_PT1000_DEF_L1},
	
	{"Thermistor.10K.A.Def.L1", 	ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_A_10K_DEF_L1},
	{"Thermistor.10K.B.Def.L1", 	ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_B_10K_DEF_L1},
	
	{"Voltage.Pressure.A.Def.L1", ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_HONEYWELL_TRUSTABILITY},
	{"Voltage.Pressure.B.Def.L1", ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_AMPHENOL_NPA300X},
	
	{"Current.Pressure.A.Def.L1", ADI_SENSE_1000_ADC_SENSOR_CURRENT_PRESSURE_HONEYWELL_PX2},
	
	{"I2C.Pressure.A.Def.L1", 	ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_HONEYWELL_HUMIDICON},
	{"I2C.Pressure.B.Def.L1", 	ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_SENSIRION_SHT3X},
	
	{"SPI.Pressure.A.Def.L1", 	ADI_SENSE_1000_SPI_SENSOR_PRESSURE_HONEYWELL_TRUSTABILITY},
	{"SPI.Accelerometer.A.Def.L1",ADI_SENSE_1000_SPI_SENSOR_ACCELEROMETER_ADI_ADXL362}
};

/*! MeasurementMaxValue Values */
static StringToStructVals measurementMaxVals[] = {
	{"N/A", 				0}
};

/*! MeasurementMinValue Values */
static StringToStructVals measurementMinVals[] = {
	{"N/A", 				0}
};

/*! Gain Values */
static StringToStructVals gainVals[] = {
	{"N/A", 				ADI_SENSE_1000_ADC_GAIN_1X},
	{"1", 				ADI_SENSE_1000_ADC_GAIN_1X},
	{"2",					ADI_SENSE_1000_ADC_GAIN_2X},
	{"4", 				ADI_SENSE_1000_ADC_GAIN_4X},
	{"8", 				ADI_SENSE_1000_ADC_GAIN_8X},
	{"16", 				ADI_SENSE_1000_ADC_GAIN_16X},
	{"32", 				ADI_SENSE_1000_ADC_GAIN_32X},
	{"64", 				ADI_SENSE_1000_ADC_GAIN_64X},
	{"128", 				ADI_SENSE_1000_ADC_GAIN_128X}
};

/*! CurrentOutput Values */
static StringToStructVals currentOutputLevelVals[] = {
	{"N/A", 				ADI_SENSE_1000_ADC_EXC_CURRENT_NONE},
	{"0.00005", 			ADI_SENSE_1000_ADC_EXC_CURRENT_50uA},
	{"0.0001", 				ADI_SENSE_1000_ADC_EXC_CURRENT_100uA},
	{"0.00025", 			ADI_SENSE_1000_ADC_EXC_CURRENT_250uA},
	{"0.0005", 				ADI_SENSE_1000_ADC_EXC_CURRENT_500uA},
	{"0.00075", 			ADI_SENSE_1000_ADC_EXC_CURRENT_750uA},
	{"0.001", 				ADI_SENSE_1000_ADC_EXC_CURRENT_1000uA}
};

/*! FilterType Values */
static StringToStructVals filterTypeVals[] = {
	{"N/A", 				ADI_SENSE_1000_ADC_FILTER_SINC4},
	{"Sinc.4", 				ADI_SENSE_1000_ADC_FILTER_SINC4},
	{"FIR.20SPS", 			ADI_SENSE_1000_ADC_FILTER_FIR_20SPS},
	{"FIR.25SPS", 			ADI_SENSE_1000_ADC_FILTER_FIR_25SPS}
};

/*! FS Values */
static StringToStructVals fsVals[] = {
	{"N/A", 				0}
};

/*! Truth table used to determine reference settings due to differences in header and config file*/
#define REF_SELECT_REFIN1		"Refin1"
#define REF_SELECT_REFIN2		"Refin2"
#define REF_SELECT_INTERNAL		"Internal"

/*! ReferenceResistorSelect Values*/
#define REF_RES_SELECT_INTERNAL	"Internal"
#define REF_RES_SELECT_EXTERNAL	"External"
#define REF_RES_SELECT_NA		"N/A"

/*! ReferenceDisable Values */
static StringToStructVals referenceDisableBufferVals[] = {
	{"N/A",     			0},
	{"False", 				0},
	{"True",  				1}
};

/*! EnableVbias Values */
static StringToStructVals enableVbiasVals[] = {
	{"N/A",     			0},
	{"False", 				0},
	{"True",  				1}
};

/*! DeviceAddress Values */
static StringToStructVals deviceAddressVals[] = {
	{"N/A",     			0}
};


/*! Enum to identify the keyword being parsed. This relates directly to keywordConvert[]*/
enum CONFIG_ATTRIBUTE {
	PRODUCT_ID = 0,
	OPERATIONAL_MODE,
	POWER_MODE,
	CYCLE_TIME,
	DATA_READY_MODE,
	GLOBAL_DIAGNOSTICS,
	MEASUREMENT_SPECIFIC_DIAGNOSTICS,
	DIAGNOSTIC_MEASUREMENT_FREQUENCY,
	
	MEASUREMENT_ENABLE,
	MEASUREMENTS_PER_CYCLE,
	SETTLING_TIME,
	SENSOR_TYPE,
	MEASUREMENT_MAX_VALUE,
	MEASUREMENT_MIN_VALUE,
	GAIN,
	VBIAS_ENABLE,
	EXCITATION_CURRENT,
	REFERENCE,
	FILTER_TYPE,
	FILTER_FS,
	ASSIGNED_COMPENSATION_MEASUREMENT_CHANNEL,    
	PUBLISH_MEASUREMENT,
	DISPLAY_UNIT,
	
	DEVICE_ADDRESS,
	//NOT NEEDED FOR FIRST SAMPLING
	/*
	READ_COMMAND,
	CONFIG_COMMAND,
	NUMBER_OF_BITS,
	FRAME_WIDTH,
	LEFT_ALIGNMENT,
	DATA_ENDIANNESS,
	OFFSET,
	CODING,
	CPOL_CPHA,
	*/
	
	//size of enum
	ENUM_ATTRIBUTE_SIZE
};


/*! Struct used to match a keyword from the config file and locate a struct array of possible values*/
static struct keywordConvert keywordConvert[] = {
	//string               				//string values to struct values//amount of possible string values
	//device wide parameters
	{"ProductID",                             	productIdVals,            	ARRAY_SIZE(productIdVals)},
	{"OperationalMode",                       	operationalModeVals,      	ARRAY_SIZE(operationalModeVals)},
	{"PowerMode",                             	powerModeVals,            	ARRAY_SIZE(powerModeVals)},
	{"CycleTime",                             	cycleTimeVals,                ARRAY_SIZE(cycleTimeVals)},
	{"DataReadyMode",                         	dataReadyModeVals,      	ARRAY_SIZE(dataReadyModeVals)},
	{"GlobalDiagnostics",					globalDiagnosticVals,		ARRAY_SIZE(globalDiagnosticVals)},
	{"MeasurementSpecificDiagnostics",			measurementDiagnosticVals,	ARRAY_SIZE(measurementDiagnosticVals)},
	{"DiagnosticMeasurementFrequency",			diagnosticFrequencyVals,	ARRAY_SIZE(diagnosticFrequencyVals)},
	
	//channel parameters
	{"MeasurementEnable",                       	enableChannelVals,        	ARRAY_SIZE(enableChannelVals)},
	{"MeasurementsPerCycle",                    	measurementsPerCycleVals,     ARRAY_SIZE(measurementsPerCycleVals)},
	{"SettlingTime",                    		settlingTimeVals,           	ARRAY_SIZE(settlingTimeVals)}, /*< Requires two arguments, see documentation */
	{"SensorType",                              	sensorVals,               	ARRAY_SIZE(sensorVals)},
	{"MeasurementMaxValue",                     	measurementMaxVals,           ARRAY_SIZE(measurementMaxVals)},
	{"MeasurementMinValue",                     	measurementMinVals,           ARRAY_SIZE(measurementMinVals)},
	{"Gain",                                    	gainVals,                 	ARRAY_SIZE(gainVals)},
	{"VBiasEnable",                             	enableVbiasVals,          	ARRAY_SIZE(enableVbiasVals)},
	{"ExcitationCurrent",                       	currentOutputLevelVals,   	ARRAY_SIZE(currentOutputLevelVals)},
	{"Reference",                 			0,      				0}, /*< Requires four arguments, see documentation. Handled by #define truth table */
	{"FilterType",                              	filterTypeVals,           	ARRAY_SIZE(filterTypeVals)},
	{"FS",                                      	fsVals,                     	ARRAY_SIZE(fsVals)},
	{"AssignedCompensationMeasurementChannel",  	channelIdVals,            	ARRAY_SIZE(channelIdVals)},
	{"PublishMeasurement",                      	publishMeasurementVals,    	ARRAY_SIZE(publishMeasurementVals)},
	{"DisplayUnit",                             	unitTranslationVals,      	ARRAY_SIZE(unitTranslationVals)},
	
	//LUT and COEFF Table input here
	
	//digital parameters
	{"DeviceAddress",          				deviceAddressVals,		ARRAY_SIZE(deviceAddressVals)}
	//NOT NEEDED FOR FIRST SAMPLING
	/*
	{"ReadCommand",                 			NULL,                  		0},
	{"ConfigCommand",                 			NULL,                  		0},
	{"NumberOfBits",                 			NULL,                  		0},
	{"FrameWidth",                 			NULL,                  		0},
	{"LeftAlignment",                 			NULL,                  		0},
	{"DataEndianness",                 			NULL,                  		0},
	{"Offset",                 				NULL,                  		0},
	{"Coding",                 				NULL,                  		0},
	{"CPOL_CPHA",                 			NULL,                  		0}
	*/
};


#endif

