/*!
 ******************************************************************************
 * @file:   adi_sense_1000_sensor_types.h
 * @brief:  Sensor type definitions for ADI Sense 1000.
 *-----------------------------------------------------------------------------
 */

/*
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef __ADI_SENSE_1000_SENSOR_TYPES_H__
#define __ADI_SENSE_1000_SENSOR_TYPES_H__

/*! @addtogroup ADI_Sense_1000_Api ADI Sense 1000 Host Library API
 *  @{
 */

#ifdef __cplusplus
extern "C" {
#endif

/*! ADI Sense 1000 measurement channel identifiers */
typedef enum {
    ADI_SENSE_1000_CHANNEL_ID_NONE  = -1,
    /*!< Used to indicate when no channel is selected (e.g. compensation channel) */

    ADI_SENSE_1000_CHANNEL_ID_CJC_0 = 0,
    /*!< Cold-Juction Compensation channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_CJC_1,
    /*!< Cold-Juction Compensation channel #1 */
    ADI_SENSE_1000_CHANNEL_ID_SENSOR_0,
    /*!< Analog Sensor channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_SENSOR_1,
    /*!< Analog Sensor channel #1 */
    ADI_SENSE_1000_CHANNEL_ID_SENSOR_2,
    /*!< Analog Sensor channel #2 */
    ADI_SENSE_1000_CHANNEL_ID_SENSOR_3,
    /*!< Analog Sensor channel #3 */
    ADI_SENSE_1000_CHANNEL_ID_VOLTAGE_0,
    /*!< Analog 0-10V Voltage Sensor channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_CURRENT_0,
    /*!< Analog 4-20mA Current Sensor channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_I2C_0,
    /*!< Digital I2C Sensor channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_I2C_1,
    /*!< Digital I2C Sensor channel #1 */
    ADI_SENSE_1000_CHANNEL_ID_SPI_0,
    /*!< Digital SPI Sensor channel #0 */
    ADI_SENSE_1000_CHANNEL_ID_SPI_1,
    /*!< Digital SPI Sensor channel #1 */
    ADI_SENSE_1000_CHANNEL_ID_SPI_2,
    /*!< Digital SPI Sensor channel #2 */

    ADI_SENSE_1000_MAX_CHANNELS,
    /*!< Maximum number of measurement channels on the ADI Sense 1000 */
} ADI_SENSE_1000_CHANNEL_ID;

/*! ADI Sense 1000 analog sensor type options
 *
 *  Select the sensor type that is connected to an ADC analog measurement
 *  channel.
 *
 *  @note Some channels may only support a subset of the available sensor types
 *        below.
 *
 *  @note The sensor type name may include a classification suffix:
 *        - _DEF_L1: pre-defined sensor using built-in linearisation data
 *        - _DEF_L2: pre-defined sensor using user-supplied linearisation data
 *        Where the suffix is absent, assume the _DEF_L1 classification above.
 */
typedef enum {
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_T_DEF_L1 = 0,
    /*!< Standard T-type Thermocouple temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_J_DEF_L1 = 1,
    /*!< Standard J-type Thermocouple temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_K_DEF_L1 = 2,
    /*!< Standard K-type Thermocouple temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_1_DEF_L2 = 12,
    /*!< Standard thermocouple temperature sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_2_DEF_L2 = 13,
    /*!< Standard thermocouple temperature sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_3_DEF_L2 = 14,
    /*!< Standard thermocouple temperature sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_4_DEF_L2 = 15,
    /*!< Standard thermocouple temperature sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_T_ADV_L1 = 16,
    /*!< T-type thermocouple temperature sensor with default linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_K_ADV_L1 = 17,
    /*!< T-type thermocouple temperature sensor with default linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_J_ADV_L1 = 18,
    /*!< T-type thermocouple temperature sensor with default linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_1_ADV_L2 = 28,
    /*!< Thermocouple temperature sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_2_ADV_L2 = 29,
    /*!< Thermocouple temperature sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_3_ADV_L2 = 30,
    /*!< Thermocouple temperature sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMOCOUPLE_4_ADV_L2 = 31,
    /*!< Thermocouple temperature sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_PT100_DEF_L1 = 32,
    /*!< Standard 2-wire PT100 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_PT1000_DEF_L1 = 33,
    /*!< Standard 2-wire PT1000 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_1_DEF_L2 = 44,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_2_DEF_L2 = 45,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_3_DEF_L2 = 46,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_4_DEF_L2 = 47,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_PT100_ADV_L1 = 48,
    /*!< Standard 2-wire PT100 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_PT1000_ADV_L1 = 49,
    /*!< Standard 2-wire PT1000 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_1_ADV_L2 = 60,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_2_ADV_L2 = 61,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_3_ADV_L2 = 62,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_2WIRE_4_ADV_L2 = 63,
    /*!< 2-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Cold-Juction Compensation and Analog Sensor channels
     *       only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_PT100_DEF_L1 = 64,
    /*!< Standard 3-wire PT100 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_PT1000_DEF_L1 = 65,
    /*!< Standard 3-wire PT1000 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_1_DEF_L2 = 76,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_2_DEF_L2 = 77,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_3_DEF_L2 = 78,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_4_DEF_L2 = 79,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_PT100_ADV_L1 = 80,
    /*!< Standard 3-wire PT100 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_PT1000_ADV_L1 = 81,
    /*!< Standard 3-wire PT1000 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_1_ADV_L2 = 92,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_2_ADV_L2 = 93,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_3_ADV_L2 = 94,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_3WIRE_4_ADV_L2 = 95,
    /*!< 3-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_PT100_DEF_L1 = 96,
    /*!< Standard 4-wire PT100 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_PT1000_DEF_L1 = 97,
    /*!< Standard 4-wire PT1000 RTD temperature sensor with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_1_DEF_L2 = 108,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_2_DEF_L2 = 109,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_3_DEF_L2 = 110,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_4_DEF_L2 = 111,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_PT100_ADV_L1 = 112,
    /*!< Standard 4-wire PT100 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_PT1000_ADV_L1 = 113,
    /*!< Standard 4-wire PT1000 RTD temperature sensor with default
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_1_ADV_L2 = 124,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_2_ADV_L2 = 125,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_3_ADV_L2 = 126,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_RTD_4WIRE_4_ADV_L2 = 127,
    /*!< 4-wire RTD temperature sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_A_10K_DEF_L1 = 128,
    /*!< Standard 10kOhm NTC Thermistor temperature sensor with Steinhart–Hart
     *   linearisation equation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_B_10K_DEF_L1 = 129,
    /*!< Standard 10kOhm NTC Thermistor temperature sensor with Beta
     *   linearisation equation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_1_DEF_L2 = 140,
    /*!< Thermistor sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_2_DEF_L2 = 141,
    /*!< Thermistor sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_3_DEF_L2 = 142,
    /*!< Thermistor sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_4_DEF_L2 = 143,
    /*!< Thermistor sensor with user-defined linearisation and
     *   default configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_A_10K_ADV_L1 = 144,
    /*!< 10kOhm NTC Thermistor temperature sensor with Steinhart–Hart
     *   linearisation equation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_B_10K_ADV_L1 = 145,
    /*!< 10kOhm NTC Thermistor temperature sensor with Beta
     *   linearisation equation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_1_ADV_L2 = 156,
    /*!< Thermistor sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_2_ADV_L2 = 157,
    /*!< Thermistor sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_3_ADV_L2 = 158,
    /*!< Thermistor sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_THERMISTOR_4_ADV_L2 = 159,
    /*!< Thermistor sensor with user-defined linearisation and
     *   advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_1_DEF_L2 = 160,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_2_DEF_L2 = 161,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_3_DEF_L2 = 162,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_4_DEF_L2 = 163,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_1_ADV_L2 = 176,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_2_ADV_L2 = 177,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_3_ADV_L2 = 178,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_4WIRE_4_ADV_L2 = 179,
    /*!< Standard 4-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_1_DEF_L2 = 192,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_2_DEF_L2 = 193,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_3_DEF_L2 = 194,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_4_DEF_L2 = 195,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_1_ADV_L2 = 208,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_2_ADV_L2 = 209,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_3_ADV_L2 = 210,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_BRIDGE_6WIRE_4_ADV_L2 = 211,
    /*!< Standard 6-wire Bridge Transducer sensor with user-defined
     *   linearisation and advanced configuration options
     *
     * @note For use with Analog Sensor channels only
     * @note Bridge Excition Voltage must be selected as reference
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE = 256,
    /*!< Generic voltage sensor with no linearisation applied
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_HONEYWELL_TRUSTABILITY = 272,
    /*!< Honeywell Pressure voltage sensor (HSCMRNN1.6BAAA3) with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_AMPHENOL_NPA300X = 273,
    /*!< Amphenol Pressure voltage sensor (NPA-300B-015A) with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_VOLTAGE_PRESSURE_3_DEF = 274,
    /*!< Generic pressure voltage sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog 0-10V Voltage Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_CURRENT = 384,
    /*!< Generic current sensor with no linearisation applied
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_CURRENT_PRESSURE_HONEYWELL_PX2 = 385,
    /*!< Honeywell Pressure current sensor (PX2CN2XX100PACH) with default
     *   linearisation and default configuration options
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
    ADI_SENSE_1000_ADC_SENSOR_CURRENT_PRESSURE_2_DEF = 386,
    /*!< Generic pressure current sensor with user-defined
     *   linearisation and default configuration options
     *
     * @note For use with Analog 4-20mA Current Sensor channels only
     */
} ADI_SENSE_1000_ADC_SENSOR_TYPE;

/*! ADI Sense 1000 I2C digital sensor type options
 *
 *  Select the sensor type that is connected to an I2C digital measurement
 *  channel.
 *
 *  @note These are pre-defined sensors using built-in linearisation data
 */
typedef enum {
    ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_HONEYWELL_HUMIDICON = 2112,
    /*!< Honeywell HiH9000-series humidity sensor with default linearisation
     *   and default configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
    ADI_SENSE_1000_I2C_SENSOR_HUMIDITY_SENSIRION_SHT3X = 2113,
    /*!< Sensirion SHT35-DIS-B humidity sensor with default linearisation
     *   and default configuration options
     *
     * @note For use with I2C Digital Sensor channels only
     */
} ADI_SENSE_1000_I2C_SENSOR_TYPE;

/*! ADI Sense 1000 SPI digital sensor type options
 *
 *  Select the sensor type that is connected to an SPI digital measurement
 *  channel.
 *
 *  @note These are pre-defined sensors using built-in linearisation data
 */
typedef enum {
    ADI_SENSE_1000_SPI_SENSOR_PRESSURE_HONEYWELL_TRUSTABILITY = 3072,
    /*!< Honeywell HSCDRNN1.6BASA3 pressure sensor with default linearisation
     * and default configuration options
     *
     * @note For use with SPI Digital Sensor channels only
     */
    ADI_SENSE_1000_SPI_SENSOR_ACCELEROMETER_ADI_ADXL362 = 3200,
    /*!< Analog Devices ADxL362 3-axis accelerometer sensor with default
     *   linearisation and default configuration options(*)
     *
     * @note (*) Custom configuration command can be optionally specified
     *
     * @note For use with SPI Digital Sensor channels only
     *
     * @note This sensor requires the use of 3 SPI Digital Sensor channels, with
     *       the sensor measurements from the X/Y/Z axes each output on a
     *       seperate dedicated channel (SPI#0/SPI#1/SPI#2, respectively)
     */
} ADI_SENSE_1000_SPI_SENSOR_TYPE;

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADI_SENSE_1000_SENSOR_TYPES_H__ */

