/*!
 ******************************************************************************
 * @file:   adi_sense_spi.cpp
 * @brief:  ADI Sense OS-dependent wrapper layer for SPI interface
 *-----------------------------------------------------------------------------
 */

/******************************************************************************
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *****************************************************************************/

#include <mbed.h>

#include "inc/adi_sense_spi.h"
#include "inc/adi_sense_log.h"

#define ADI_SENSE_SPI_MODE       0 /* CPOL=0, CPHA=0 */
#define ADI_SENSE_SPI_FRAME_SIZE 8 /* 8-bit frame size */

class SpiContext
{
public:
    SpiContext(
        PinName mosiPin,
        PinName misoPin,
        PinName sckPin,
        PinName csPin,
        unsigned maxSpeed);

    int transfer(
        void *pTxData,
        void *pRxData,
        unsigned nLength,
        bool bCsHold);

private:
    SPI _spi;
    DigitalOut _cs;

    event_callback_t _callback;
    volatile int     _cbEvent;
    volatile bool    _cbFired;

    void _cbHandler(
        int event);

    int _waitForCallback(
        void);
};

SpiContext::SpiContext(PinName mosiPin,
                       PinName misoPin,
                       PinName sckPin,
                       PinName csPin,
                       unsigned maxSpeed)
    : _spi(mosiPin, misoPin, sckPin),
      _cs(csPin, 1)
{
    _cbEvent = 0;
    _cbFired = false;
    _callback.attach(this, &SpiContext::_cbHandler);

    _spi.format(ADI_SENSE_SPI_FRAME_SIZE, ADI_SENSE_SPI_MODE);
    _spi.frequency(maxSpeed);
}

void SpiContext::_cbHandler(int event)
{
    _cbEvent = event;
    _cbFired = true;
}

int SpiContext::_waitForCallback(void)
{
    int rc = 0;

    while ((_cbFired != true) && (_cbEvent != SPI_EVENT_COMPLETE))
    { ; }

    _cbFired = false;
    _cbEvent = 0;

    return rc;
}

int SpiContext::transfer(
    void *pTxData,
    void *pRxData,
    unsigned nLength,
    bool bCsHold)
{
    int rc = 0;

    _cs = 0;

    rc  = _spi.transfer((uint8_t *)pTxData, nLength,
                        (uint8_t *)pRxData, nLength,
                        _callback, SPI_EVENT_COMPLETE);
    if (rc < 0)
    {
        _cs = 1;
        return rc;
    }   

    rc = _waitForCallback();
    if ((rc < 0) || !bCsHold)
        _cs = 1;

    return rc;
}

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Open the SPI interface and allocate resources
 */
ADI_SENSE_RESULT adi_sense_SpiOpen(
    ADI_SENSE_PLATFORM_SPI_CONFIG *pConfig,
    ADI_SENSE_SPI_HANDLE *phDevice)
{
    SpiContext *pCtx = new SpiContext((PinName)pConfig->mosiPin,
                                      (PinName)pConfig->misoPin,
                                      (PinName)pConfig->sckPin,
                                      (PinName)pConfig->csPin,
                                      pConfig->maxSpeedHz);
    if (!pCtx)
    {
        ADI_SENSE_LOG_ERROR("Failed to allocate memory for SPI interface");
        return ADI_SENSE_NO_MEM;
    }

    *phDevice = reinterpret_cast<ADI_SENSE_SPI_HANDLE>(pCtx);
    return ADI_SENSE_SUCCESS;
}

/*
 * Execute a bi-directional data transfer on the SPI interface
 */
ADI_SENSE_RESULT
adi_sense_SpiTransfer(
    ADI_SENSE_SPI_HANDLE hDevice,
    void *pTxData,
    void *pRxData,
    unsigned nLength,
    bool bCsHold)
{
    SpiContext *pCtx = reinterpret_cast<SpiContext *>(hDevice);

    if (pCtx->transfer(pTxData, pRxData, nLength, bCsHold) < 0)
    {
        ADI_SENSE_LOG_ERROR("Failed to complete SPI transfer");
        return ADI_SENSE_FAILURE;
    }

    return ADI_SENSE_SUCCESS;
}

/*
 * Close the SPI interface and free resources
 */
void adi_sense_SpiClose(
    ADI_SENSE_SPI_HANDLE hDevice)
{
    SpiContext *pCtx = reinterpret_cast<SpiContext *>(hDevice);

    delete pCtx;
}

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

