//BUCA2201
//GODJ2407

#include "mbed.h"
#include "rtos.h"
#include "EthernetInterface.h"
#include "Websocket.h"

LocalFileSystem local("local");

DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut ledError(LED4);

DigitalOut reset(p8);
RawSerial xbee(p13, p14);

RawSerial pc(USBTX, USBRX);

Thread* receptionThread;
Thread* processingThread;
Thread* ledThread;
Thread* errorDisplay;

Websocket* sock;

//Commande AT pour changer la pin0
char LED_Toggle[] = {0x7E, 0x00, 0x10, 0x17, 0x01,
 0x00, 0x13, 0xA2, 0x00, 0x40, 0x8B, 0x41, 0x8B,
 0x00, 0x00, 
 0x02, 0x44, 0x30, 0x00, 0x25}; 

//Trois commandes pour changer le PAN ID d'un device
char AT_ID[] = {0x7E, 0x00, 0x06, 0x09, 0x01, 0x49 , 0x44, 0x00, 0x00, 0x00}; // +DATA + CS (7 & 8)
const char AT_WR[] = {0x7E, 0x00, 0x04, 0x09, 0x01, 0x57, 0x52, 0x4C};
const char AT_AC[] = {0x7E, 0x00, 0x04, 0x09, 0x01, 0x41, 0x43, 0x71};

typedef struct {
    char msg[25];
} message;

//Storage des adresses qu'on toggle leur pin0
typedef struct {
    char adr[10];
} address;

address addresses[2];
char addressCounter = 0;

Queue<message, 25> queue;
MemoryPool<message, 25> mPool;

//URL du serveur
char url[128];

//Calcul le checksum d'une trame
char checksum(char array[], char length)
{
    char cs = 0;
    for (int i = 3; i<length - 1; i++) {
        cs += array[i];
    }
    return 0xFF - cs;
}

//Convertie les donnes de l'acc en donnees lisible
int traitementAcc(char msb, char lsb)
{
    int val = msb > 127 ? 0xFFFFF000 | (msb << 4) : msb << 4;

    // Ajout des LSB
    val += lsb >> 4;

    val = val > 2047 ? 2048 - val : val;
    val = val > 1024 ? 1024 : val;
    val = val < -1024 ? -1024 : val;

    return val;
}

void reception()
{
    while(1) {
        if(xbee.readable()) {
            message* msg = mPool.alloc();

            msg->msg[0] = xbee.getc();

            if(msg->msg[0] == 0x7E) {
                msg->msg[1] = xbee.getc();
                msg->msg[2] = xbee.getc();
                
                //On assume que la longueur est toujours plus petite que 256
                for (int i = 3; i < msg->msg[2] + 4; i++) {
                    msg->msg[i] = xbee.getc();
                    if(msg->msg[i] == 0x7E) {
                        break;//Erreur de longueur
                    }
                }

                queue.put(msg);
            } else {
                mPool.free(msg);
            }
        }
    }
}

void processing()
{
    while(1) {
        message* msg = (message*)queue.get().value.p;
        
        switch(msg->msg[3]) {
            case 0x88:
                if (msg->msg[7] != 0x00) {
                    errorDisplay->signal_set(0x1);
                    pc.printf("AT command error\r\n");
                }
                break;
            case 0x8A:
                if (msg->msg[4] != 0x00 && msg->msg[4] != 0x06) {
                    errorDisplay->signal_set(0x1);
                    pc.printf("%02x - Modem error\r\n", msg->msg[4]);
                }
                break;
            case 0x97:
                if (msg->msg[17] != 0x00) {
                    errorDisplay->signal_set(0x1);
                    pc.printf("%02x - Remote AT command error\r\n", msg->msg[4]);
                }
                break;
            case 0x90:
                if(msg->msg[2] >= 0x0C) {
                    char exists = 0;
                    
                    //Determine si c'est une nouvelle adresse
                    for (int i = 0; i< sizeof(addresses); i++) {
                        for(int j = 0; j < 10;j++) {
                            if(addresses[i].adr[j] == msg->msg[j + 4]){
                                exists++;
                            }
                        }
                        
                        if(exists == 10){ break;} //Existe deja
                        exists = 0; //Regarde les autres adresses
                    }
                    
                    if (exists < 10) {                        
                        pc.printf("New Address: ");
                        for(int i = 0; i < 10;i++){
                            addresses[addressCounter].adr[i] = msg->msg[i + 4];
                            pc.printf("%02x ", addresses[addressCounter].adr[i]);
                        }
                        pc.printf("\r\n");
                        
                        addressCounter++;
                    }

                    //Extraction des donnes pertinentes
                    char data[7];
                    for(int i = 15; i < msg->msg[2] + 3; i++) {
                        data[i - 15] = msg->msg[i];
                    }
                    
                    //Lecture du premier char pour connaitre le type
                    char reception[256];
                    switch(data[0]) {
                        case 0x00:
                            char response[] = "Etat du bouton:  ";
                            if (data[1] == 0x00) {
                                response[16] = '0';
                            } else if (data[1] == 0x01) {
                                response[16] = '1';
                            }
                            sock->send(response);
                            led2 = !led2;
                            pc.printf("Sending to server: %s\r\n", response);                            
                            
                            sock->read(reception);
                            pc.printf("Received from server: %s\r\n", reception);
                            break;
                        case 0x01:
                            int x = traitementAcc(data[1], data[2]);
                            int y = traitementAcc(data[3], data[4]);
                            int z = traitementAcc(data[5], data[6]);
                            char out[128];
                            sprintf(out, "Accelerometre [x: %d, y: %d, z: %d]", x,y,z);

                            sock->send(out);
                            led3 = !led3;
                            
                            pc.printf("Sending to server: %s\r\n", out);
                        
                            sock->read(reception);
                            pc.printf("Received from server: %s\r\n", reception);
                            break;
                    }
                }
                break;
            default:
                errorDisplay->signal_set(0x1);
                pc.printf("Invalid command error\r\n");
                break;
        }
        mPool.free(msg);
    }
}

void error_display()
{
    while(1) {
        Thread::signal_wait(0x1);
        ledError = 1;
        wait(1);
        ledError = 0;
    }
}

void flashLED()
{
    bool high = true;
    while(1) {
        Thread::signal_wait(0x1);
        
        //Itteration des adresses qu'on doit faire flasher
        for (int i = 0; i < addressCounter; i++) {            
            for (int j = 0; j < 10; j++) {
                LED_Toggle[j + 5] = addresses[i].adr[j];
            }
            
            LED_Toggle[18] = high ? 0x05 : 0x04;
            LED_Toggle[19] = checksum(LED_Toggle, sizeof(LED_Toggle));

            while(!xbee.writeable()) {}
            for (int j = 0; j < sizeof(LED_Toggle); j++) {
                xbee.putc(LED_Toggle[j]);
            }
        }
        led1 = !led1;
        high = !high;
    }
}

void LEDSignal()
{
    ledThread->signal_set(0x1);
}

int main()
{        
    reset = 0;
    wait(1);
    reset = 1;
    wait(1);

    int pan;
    
    //Lecture de la config
    FILE* f = fopen("/local/coord.cfg", "r");
    fscanf(f,"%x", &pan);
    fscanf(f,"%s", &url);
    fclose(f);

    pc.printf("URL: %s | ", url);

    //Convertie le pan ID
    char buff[2];
    buff[0]=(pan>>8)&0xff;
    buff[1]=(pan)&0xff;

    AT_ID[7] = buff[0];
    AT_ID[8] = buff[1];

    pc.printf("PAN: %02x%02x\r\n", AT_ID[7],AT_ID[8]);

    char cs = checksum(AT_ID, sizeof(AT_ID));

    AT_ID[9] = cs;
    
    //Configuration Ethernet
    EthernetInterface eth;
    eth.init(); //Use DHCP
    //eth.init("192.168.0.3", "255.255.255.0", "192.168.0.2");
    eth.connect();
    pc.printf("IP Address is %s\r\n", eth.getIPAddress());
    Websocket ws(url);
    sock = &ws;
    while(sock->connect() < 0){}//Attente du serveur
    
    //Setup du pan ID
    for(char i = 0; i<sizeof(AT_ID); i++) {
        xbee.putc(AT_ID[i]);
    }

    for(char i = 0; i<sizeof(AT_WR); i++) {
        xbee.putc(AT_WR[i]);
    }

    for(char i = 0; i<sizeof(AT_AC); i++) {
        xbee.putc(AT_AC[i]);
    }
    wait(1);

    Thread errorDisplayLocal;
    errorDisplay = &errorDisplayLocal;
    errorDisplay->start(&error_display);

    Thread localReceptionThread;
    receptionThread = &localReceptionThread;
    receptionThread->start(&reception);

    Thread localLedThread;
    ledThread = &localLedThread;
    ledThread->start(&flashLED);

    Ticker horloge;
    horloge.attach(&LEDSignal, 0.5); 
    
    Thread localProcessingThread;
    processingThread = &localProcessingThread;
    processingThread->start(&processing);
    
    while(1) {
    }
}
