#ifndef _H_SENSOR_H
#define _H_SENSOR_H

#include "Vector3.h"

/// Base class for I2C-connected sensors. Defines functionality supported by all sensors.
class Sensor
{
public:
    /// Defines protocol used to send data back to owner. Derive from this class and use Sensor.setDelegate() to receive sensor updates.
    class Delegate
    {
    public:
        /// A new sensor data frame, might be called several (hundred) times a second.
        virtual void sensorUpdate(Vector3 data) {}
    };

    virtual void setDelegate(Delegate &d) {
        delegate = &d;
    }

    /// Power on a sensor and make it ready for use.
    /// @return true if power-up was successful, false otherwise.
    virtual bool powerOn() = 0;

    /// Power off a sensor. This will generally only put the sensor into deep sleep.
    virtual void powerOff() = 0;

    virtual void start() = 0; ///< Start continuous data capture. If a delegate is set, its sensorUpdate() method will be called for each data frame.
    virtual void stop() = 0; ///< Stop capturing data.

    virtual Vector3 read() = 0; ///< Read and return instantaneous (current) sensor data. No need to start the sensor.

    Sensor() : delegate(&defaultDelegate) {}

protected:
    Delegate defaultDelegate;
    Delegate *delegate;
};

#endif//_H_SENSOR_H