/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "Logger.h"
#include "Configuration.h"
#include "BLEDevice.h"
#include "TransferService.h"

BLEDevice ble;

static Gap::ConnectionParams_t connectionParams;

static const uint8_t *uuidlist = Transfer::getServiceUUIDp();
static uint8_t uuidlistrev[16];
static const char *DEVICE_NAME = "SD";

void bluetoothInit();

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    DEBUG("Disconnected\n\r");
    ble.startAdvertising();
    DEBUG("Advertising...\r\n");
}

void onConnectionCallback(Gap::Handle_t handle, Gap::addr_type_t peerAddrType, const Gap::address_t peerAddr, const Gap::ConnectionParams_t *params)
{
    DEBUG("____[ Connected ]______________________________________\r\n");
    DEBUG("Conn. params => min=%d, max=%d, slave=%d, supervision=%d\r\n",
          params->minConnectionInterval, params->maxConnectionInterval, params->slaveLatency, params->connectionSupervisionTimeout);

    connectionParams.minConnectionInterval        = Config::minConnectionInterval;
    connectionParams.maxConnectionInterval        = Config::maxConnectionInterval;
    connectionParams.slaveLatency                 = Config::slaveLatency;
    connectionParams.connectionSupervisionTimeout = Config::supervisionTimeout;
    if (ble.updateConnectionParams(handle, &connectionParams) != BLE_ERROR_NONE) {
        DEBUG("failed to update connection paramter\r\n");
    }

    Transfer::reset();
}

void onUpdatesEnabled(Gap::Handle_t handle)
{
    DEBUG("Notifications enabled for %d\r\n", handle);
}

void onDataWritten(const GattCharacteristicWriteCBParams *params)
{
    // bubble up to services, they will emit callbacks if handle matches
    Transfer::handleDataWritten(params);
}

void bluetoothInit()
{
    DEBUG("Bluetooth initialising...\r\n");
    ble.init();
    ble.setDeviceName(Config::deviceName);
    ble.onDisconnection(disconnectionCallback);
    ble.onConnection(onConnectionCallback);
    ble.onDataWritten(onDataWritten);
    ble.onUpdatesEnabled(onUpdatesEnabled);

    // Make sure we use our preferred conn. parameters
    connectionParams.minConnectionInterval        = Config::minConnectionInterval;
    connectionParams.maxConnectionInterval        = Config::maxConnectionInterval;
    connectionParams.slaveLatency                 = Config::slaveLatency;
    connectionParams.connectionSupervisionTimeout = Config::supervisionTimeout;
    ble.setPreferredConnectionParams(&connectionParams);
    ble.getPreferredConnectionParams(&connectionParams);
    DEBUG("Conn. params => min=%d, max=%d, slave=%d, supervision=%d\r\n",
          connectionParams.minConnectionInterval,
          connectionParams.maxConnectionInterval,
          connectionParams.slaveLatency,
          connectionParams.connectionSupervisionTimeout);

    // Initialise transfer service
    Transfer::init(ble);

    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS, uuidlistrev, 16);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.setAdvertisingInterval(Config::advertisingInterval);
    ble.startAdvertising();
    DEBUG("Ready. Advertising.\r\n");
}

int main(void)
{
    DEBUG("Initialising TRANSFER PRO | Built %s %s\n\r", __DATE__, __TIME__);

    for (int i = 0; i < 16; i++) {
        uuidlistrev[15 - i] = uuidlist[i];
    }

    bluetoothInit();

    while (true) {
        ble.waitForEvent();
    }
}
