/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "BLEDevice.h"
#include "UARTService.h"

#include "SharpLCD.hpp"
#include "font.h"

BLEDevice  ble;
DigitalOut led1(LED1);

bool rxPayloadUpdated = false;
const static unsigned MAX_SIZEOF_RX_PAYLOAD = 20;
char rxPayload[MAX_SIZEOF_RX_PAYLOAD] = {0,};

uint8_t framebuffer[SharpLCD::SIZEOF_FRAMEBUFFER_FOR_ALLOC];

UARTService *uartServicePtr;

const char *deviceName = "lcdDemo";
SharpLCD lcd(p0 /* display enable */, SPI_PSELSS0, SPI_PSELMOSI0, SPI_PSELMISO0, SPI_PSELSCK0, NC);
SharpLCD::FrameBuffer fb(framebuffer);

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    ble.startAdvertising();
}

void onDataWritten(const GattCharacteristicWriteCBParams *params)
{
    if ((uartServicePtr != NULL) && (params->charHandle == uartServicePtr->getTXCharacteristicHandle())) {
        uint16_t bytesRead = params->len;
        if (bytesRead < MAX_SIZEOF_RX_PAYLOAD) {
            strncpy(rxPayload, (const char *)params->data, MAX_SIZEOF_RX_PAYLOAD - 1);
            rxPayload[bytesRead] = '\0';
            rxPayloadUpdated = true;
            ble.updateCharacteristicValue(uartServicePtr->getRXCharacteristicHandle(), params->data, bytesRead);
        }
    }
}

void periodicCallback(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */
}

int main(void)
{
    led1 = 1;
    Ticker ticker;
    ticker.attach(periodicCallback, 1);

    ble.init();
    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(onDataWritten);

    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME, (const uint8_t *)deviceName, strlen(deviceName));

    ble.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble.startAdvertising();

    UARTService uartService(ble);
    uartServicePtr = &uartService;

    lcd.enableDisplay();
    lcd.clear();
    fb.printString(lookupFontFace("DejaVu Serif", 8), 20, 40, BLACK, "Init");
    lcd.drawFrameBuffer(fb);

    while (true) {
        if (rxPayloadUpdated) {
            fb.clear();
            lcd.drawFrameBuffer(fb);
            fb.printString(lookupFontFace("DejaVu Serif", 8), 20, 40, BLACK, (const char *)rxPayload);
            lcd.drawFrameBuffer(fb);

            rxPayloadUpdated = false;
        }

        ble.waitForEvent();
    }
}
