/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/BLE.h"
#include "EddystoneService.h"

EddystoneService *eddyServicePtr;

void onBleInitError(BLE::InitializationCompleteCallbackContext* initContext)
{
    /* Initialization error handling goes here... */
    (void) initContext;
}

/*
 * Update TLM frame battery voltage value.
 */
uint16_t tlmBatteryVoltageCallback(uint16_t prevBatteryVoltage)
{
    prevBatteryVoltage++;
    return prevBatteryVoltage;
}

/*
 * Update TLM frame beacon temperature value.
 */
uint16_t tlmBeaconTemperatureCallback(uint16_t prevBeaconTemperature)
{
    prevBeaconTemperature++;
    return prevBeaconTemperature;
}

void bleInitComplete(BLE::InitializationCompleteCallbackContext* initContext)
{
    BLE         &ble  = initContext->ble;
    ble_error_t error = initContext->error;

    if (error != BLE_ERROR_NONE) {
        onBleInitError(initContext);
        return;
    }

    /* Set UID and TLM frame data */
    const UIDNamespaceID_t uidNamespaceID = {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99};
    const UIDInstanceID_t  uidInstanceID  = {0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF};
    uint8_t tlmVersion = 0x00;

    /* Initialize a EddystoneBeaconConfig service providing config params, default URI, and power levels. */
    static const PowerLevels_t defaultAdvPowerLevels = {-47, -33, -21, -13}; // Values for ADV packets related to firmware levels, calibrated based on measured values at 1m
    static const PowerLevels_t radioPowerLevels      = {-30, -16, -4, 4};    // Values for radio power levels, provided by manufacturer.

    /* Set everything to defaults */
    eddyServicePtr = new EddystoneService(ble, defaultAdvPowerLevels, radioPowerLevels);

    /* Set default URL, UID and TLM frame data if not initialized through the config service */
    eddyServicePtr->setURLData("http://mbed.org");
    eddyServicePtr->setUIDData(&uidNamespaceID, &uidInstanceID);
    eddyServicePtr->setTLMData(tlmVersion);

    /* Set battery voltage and temperature callbacks */
    eddyServicePtr->onTLMBatteryVoltageUpdate(tlmBatteryVoltageCallback);
    eddyServicePtr->onTLMBeaconTemperatureUpdate(tlmBeaconTemperatureCallback);

    /* Start Eddystone in config mode */
    eddyServicePtr->startBeaconService(5, 5, 5);
}

int main(void)
{
    BLE& ble = BLE::Instance(BLE::DEFAULT_INSTANCE);
    ble.init(bleInitComplete);

    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while (ble.hasInitialized()  == false) { /* spin loop */ }
    
    while (true) {
        ble.waitForEvent(); /* this will return upon any system event (such as an interrupt or a ticker wakeup) */
    }
}
