/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "AltBeaconService.h"
#include "ble/BLE.h"

/**
 * For this demo application, populate the beacon advertisement payload
 * with 2 AD structures: FLAG and MSD (manufacturer specific data).
 *
 * Reference:
 *  Bluetooth Core Specification 4.0 (Vol. 3), Part C, Section 11, 18
 */

/**
 * The AltBeacon requires a manufacturer ID, and a Beacon ID
 * the first 16 bytes of the BeaconID should be a UUID and the remaining
 * 4 bytes can be used as you see fit.
 *
 * Note: please remember to calibrate your beacon
 * RSSI for more accurate results.
 */
uint8_t beaconID[]      = {0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                           0x10,0x11,0x12,0x13,0x14,0x15,0x00,0x01,0x00,0x02};
uint16_t manufacturerID = 0x5900; /* Nordic SIG ID */
int8_t   rssi           = -122;

AltBeaconService *altBeaconServicePtr;

/**
 * This function is called when the ble initialization process has failed
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
    /* Initialization error handling should go here */
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    /* Initialize AltBeacon */
    altBeaconServicePtr =new AltBeaconService(ble, manufacturerID, beaconID, rssi);

    /* Set advertising time */
    ble.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    
    /* Start advertising */
    ble.startAdvertising();
}

int main(void)
{
    BLE& ble = BLE::Instance();

    /* Initialize BLE baselayer */ 
    ble.init(bleInitComplete);

    while(true) {
        ble.waitForEvent(); /* Allow low power operation */
    }
}
