/* mbed Microcontroller Library
 * Copyright (c) 2006-2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#ifndef __BLE_ALTBEACON_SERVICE_H__
#define __BLE_ALTBEACON_SERVICE_H__

#include "ble/BLE.h"

/**
* @class AltBeaconService
* @brief AltBeacon Service. This service sets up a device to broadcast advertising packets to mimic an AltBeacon<br>
*/

class AltBeaconService
{
public:
    /**
    * @param[in] _ble
    *               BLE object for the underlying controller.
    * @param[in] mfgID
    *               The beacon device manufacturer's company identifier code. 
    *               Usually this will coorespond to the companies BLE SIG assigned number.
    * @param[in] beaconID
    *               A 20-byte value uniquely identifying the beacon. 
    *               The big endian representation of the beacon identifier. 
    *               For interoperability purposes, the first 16+ bytes of the beacon 
    *               identifier should be unique to the advertiser's organizational unit. 
    *               Any remaining bytes of the beacon identifier may be subdivided as needed for the use case.
    * @param[in] refRSSI
    *               The RSSI of the beacon (as signed value from 0 to -127) as measured 1 meter from the device. Used for micro-location.
    * @param[in] mfgReserved
    *               Used for special manufacturer data. Defaults to 0x00 if not specified.
    */
    AltBeaconService(BLE &_ble, uint16_t mfgID, uint8_t beaconID[20], int8_t refRSSI, uint8_t mfgReserved = 0x00):
        ble(_ble)
    {
        /* refRSSI can only be 0 to -127, smash everything above 0 to zero */
        if (refRSSI > 0) {
            refRSSI = 0;
        }
        data.mfgID       = ((mfgID<<8) | (mfgID >>8));
        data.refRSSI     = refRSSI;
        data.beaconCode  = 0xACBE;
        data.mfgReserved = mfgReserved;

        /* copy across beacon ID */
        for(int x = 0; x < sizeof(data.beaconID); x++) {
            data.beaconID[x] = beaconID[x];
        }

        /* Set up alt beacon */
        ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
        /* Generate the 0x1BFF part of the Alt Prefix */
        ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, data.raw, sizeof(data.raw));

        /* Set advertising type */
        ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_NON_CONNECTABLE_UNDIRECTED);
    }

public:
    union {
        uint8_t raw[26]; // AltBeacon advertisment data
        struct {
            uint16_t mfgID;         // little endian representation of manufacturer ID
            uint16_t beaconCode;    // Big Endian representation of 0xBEAC
            uint8_t  beaconID[20];  // 20byte beacon ID, usually 16byte UUID w/ remainder used as necessary
            int8_t   refRSSI;       // 1 byte signed data, 0 to -127
            uint8_t  mfgReserved;   // reserved for use by manufacturer to implement special features
        };
    } data;

private:
    BLE &ble;

};

#endif /* __BLE_ALTBEACON_SERVICE_H__ */
