#pragma once

#include "mbed.h"
#include <string>

// Uncomment this in the adafruit config.h to turn off the builtin splash
//#define NO_SPLASH_ADAFRUIT
// Uncomment this in the adafruit config.h to enable all functionality
//#define GFX_WANT_ABSTRACTS

#include "Adafruit_SSD1306.h"


class Screen
{
private:
    I2C i2c;                // Communicates to the OLED screen
    Adafruit_SSD1306_I2c oled;
public:
    // The screen we use doesn't have a reset pin, but it needs one anyway.
    // Use a pin we don't use.
    Screen(PinName sda, PinName scl, PinName reset):
        i2c(sda, scl),
        oled(i2c, reset) {
            
        // According to the spec the max bitrate for the SSD1308 is 400 kbit/s
        i2c.frequency(400000);

        clear_display();
    }
    
    void clear_display() {
        oled.clearDisplay();
        oled.setTextCursor(0, 0);
        display();
    }

    void display() {
        // Apparently the display only updates every 2 "display" calls.
        // This is a bug in the library.
        oled.display();
        oled.display();
    }
    
    // State names can be a maximum of 17 characters long.
    void display_state_name(string name) {
        // Clear a previous title.
        oled.setTextCursor(0, 0);
        oled.printf("                     ");
        oled.setTextCursor(0, 0);
        oled.printf("- ");
        oled.printf(name.c_str());
        oled.printf(" -\r\n");
        display();
    }
    
    void display_emg_state(bool left_tensed, bool right_tensed) {
        oled.setTextCursor(0, 8);
        if (left_tensed) {
            oled.printf("Left  ");
        } else {
            oled.printf("      ");
        }
        if (right_tensed) {
            oled.printf("Right");
        } else {
            oled.printf("     ");
        }
    }

    // Returns a direct handle to the screen.
    // Be carefull while using the handle.
    Adafruit_SSD1306_I2c* get_screen_handle() {
        return &oled;
    }
    
    void display_up_down_arrow(bool is_up) {
        int size_x = 20;
        int size_y = 10;
        int x_start = oled.width() - size_x;
        int x_end = x_start + size_x;
        int y_start = oled.height() - size_y;
        int y_end = y_start + size_y;
        
        oled.fillRect(x_start, y_start, size_x, size_y, BLACK);
        
        if (is_up) {
            oled.fillTriangle(x_start, y_end, x_end, y_end, x_start + size_x/2, y_start, WHITE);
        } else {
            oled.fillTriangle(x_start + size_x/2, y_end, x_start, y_start, x_end, y_start, WHITE);
        }
        
        display();
    }
    
    void display_left_right_arrow(bool is_right) {
        int size_x = 20;
        int size_y = 10;
        int x_start = oled.width() - size_x * 2;
        int x_end = x_start + size_x;
        int y_start = oled.height() - size_y;
        int y_end = y_start + size_y;
        
        oled.fillRect(x_start, y_start, size_x, size_y, BLACK);     // Deletes triangle in order to make a new one
        
        if (is_right) {
            oled.fillTriangle(x_start, y_end, x_end, y_start + size_y/2, x_start, y_start, WHITE);
        } else {
            oled.fillTriangle(x_end, y_start, x_start, y_start + size_y/2, x_end, y_end, WHITE);
        }
        
        display();
    }
    
};