#include "wifi.h"
#include "utils.h"

#define BAUDR 460800
//#define BAUDR 1100000

Wifi::Wifi (PinName tx, PinName rx, PinName cts, PinName reset): 
        wifi_(tx, rx) {    
    // mbed only has CTS available, communication not working properly
    //wifi_.set_flow_control(SerialBase::CTS, cts);
    wifi_.baud(BAUDR);
}

/* 
* Initializes wifly, serves as constructor, were changing PLL speed dynamically
*
*/
void Wifi::init(void)
{
  //wifi_.baud(BAUDR);  
}

/*
* Format: (int) -1 (int) -1 (int) swimmer_id
*   many times (int) count 3*(short) acc 3*(float)gyro
*         (int) -1 (int) -1    up to 1023 zeros   
*/
int Wifi::sendFile(const char *fname)
{
    char c;
    int last = 1;

    FILE *file = fopen(fname, "rb");
    if (file == NULL) {
        printf("Unable to open %s\r\n", fname);
        return -1;
    }
    
    fseek(file, 0, SEEK_END);                           // seek to end of file
    int size = ftell(file);                             // get current file pointer
    fseek(file, 0, SEEK_SET);                           // seek back to beginning of file
    
    wifi_.putc('#');                                    // send ACK
    wifi_.putc('A');
    for(int i = 0; i < FRAME_HEADER_LEN - 2; i++)       // send num of bytes in the message
    {
        wifi_.putc((char)((size >> (i * 8)) & 0xFF));
    }
    
    printf("Ack sended...\r\n");
    printf("Transfer of %d bytes initialized...\r\n", size);
    
    if(size == 0)                                       // when nothing to send, terminated here
    {
        return 0;
    }
    
    in_buf = 0;
    while (fread(&c, sizeof(char), 1, file) != 0) {
        if ((last = bufferSendVerify(c)) == -1) {
            fclose(file);
            return -1;
        }
    }
    fclose(file);
    
    if (bufferFlush() == -1) {
        return -1;
    }

    return 0;
}

/*
 * Returns length of command data received, load the command into the cmd array and the data into the data array.
 */
int Wifi::getCmd(char* cmd, char* data)
{
    short dataLength = 0u;
    cmd[0] = '#';
    cmd[1] = 'N';
    
    /* Read the command */
    if (wifi_.readable() && wifi_.getc() == '#')
    {
        for(int i = 1; i < FRAME_HEADER_LEN; i++)
        {
            wait_ms(20);
            if (wifi_.readable())
            {
                cmd[i] = wifi_.getc();
            }
            else
            {
                cmd[1] = 'N';
            }    
        }
    }
    
    /* Read count of data */
    if(cmd[1] == 'N')
    {
        return 0;
    }
    
    for(int i = 0; i < FRAME_HEADER_LEN - 2; i++)
    {
        dataLength += (int)cmd[i+2] << (i * 8); 
    }
    
    /* Read data */
    for(int i = 0; i < dataLength; i++)
    {
        wait_ms(20);
        if (wifi_.readable())
        {
            data[i] = wifi_.getc();
        }
    }
    
    return dataLength;
}

/*
 * Sends given byte to wifly
 */
void Wifi::sendByte(char byte)
{
    // Send byte
    wifi_.putc(byte);
}

/*
 * Reads byte from wifly
 */
char Wifi::readByte(void)
{
    // Read byte
    return wifi_.getc();
}

/*
 * Returns true if some bytes are in the buffer
 */
bool Wifi::readable(void)
{
    if (wifi_.readable())
    {
        return true;    
    }
    else
    {
        return false;    
    }    
}

/*
 * Sends ACK to wiflys UART
 */
void Wifi::sendAck(void)
{
    wifi_.putc('#');
    wifi_.putc('A');
    for(int i = 0; i < FRAME_HEADER_LEN - 2; i++)
    {
        wifi_.putc(0x0);
    }
}

/*
 * Sends NACK to wiflys UART
 */
void Wifi::sendNack(void)
{
    wifi_.putc('#');
    wifi_.putc('N');
    for(int i = 0; i < FRAME_HEADER_LEN - 2; i++)
    {
        wifi_.putc(0x0);
    }
}

/*
 * Sends NACK to wiflys UART
 */
void Wifi::sendFail(void)
{
    wifi_.putc('#');
    wifi_.putc('F');
    for(int i = 0; i < FRAME_HEADER_LEN - 2; i++)
    {
        wifi_.putc(0x0);
    }
}

int Wifi::waitForAck()
{
    char cmd[FRAME_HEADER_LEN];
    char data[256];

    for (int i=0; i<64; i++)
    {
        getCmd(cmd, data);
        
        if ('A' == cmd[1])
        {
            return 1;   
        }
        else if ('F' == cmd[1])
        {
            return 0;
        }
        else
        {
            wait_ms(5);   
        }    
    }
    return -1;
    
    //if (wifi_.getc() == 'A')
    //    break;
}

/* ********************************
 * private
 * *******************************/
 
void Wifi::bufferSend(size_t size)
{
    int i;
    for (i = 0; i < size; i++) {
        wifi_.putc(buffer[i]);
    }
}

int Wifi::bufferFlush()
{
    LPC_WDT->WDFEED = 0xAA;
    LPC_WDT->WDFEED = 0x55;
    int counter = 0;
    int result = 0;
    
    if (in_buf == 0)
    {
        return 1;
    }
       
    while (counter++ < 10) 
    {
        bufferSend(in_buf);
        result = waitForAck();
                
        if(result == 1)
        {
            /* Ack - break, no need to send it again */
            break;
        }
        else if (result == -1)
        {
            /* Fail - transmition faild, terminate sending */
            return -1;
        }
        else
        {
            /* Nack - send it again*/
        }
    }
    
    if (counter >= 10)
    {
        return -1;
    }
    else
    {
        return 0;
    }
}

int Wifi::bufferSendVerify(char c)
{
    LPC_WDT->WDFEED = 0xAA;
    LPC_WDT->WDFEED = 0x55;
    int counter = 0;
    int result = 0;
    buffer[in_buf] = c;
    in_buf++;
        
    if (in_buf != SEND_SIZE)
    {
        return 1;
    }
    
    in_buf = 0;
    while (counter++ < 16) 
    {
        bufferSend(SEND_SIZE);
        result = waitForAck();
        
        if(result == 1)
        {
            /* Ack - break, no need to send it again */
            break;
        }
        else if (result == -1)
        {
            /* Fail - transmition faild, terminate sending */
            return -1;
        }
        else
        {
            /* Nack - send it again*/
        }
    }
    
    if (counter >= 10)
    {
        return -1;
    }
    else
    {
        return 0;
    }
}