/* mbed library for the mbed Lab Board  128*32 pixel LCD
 * use C12832 controller
 * Copyright (c) 2012 Peter Drescher - DC2PD
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * Library überarbeitet: 15.1.2014 Copyright (c) Dr.C.Eichtinger
 * File GraphicDisplay und TextDisplay, sowie small_7 font gestrichen
 * C12832.h und C12832.cpp überarbeitet und mit Ziffernfont ergänzt
 * Erweiterung: Zeichengröße 1..4
 */

#include "mbed.h"
#include "C12832.h"
#include <stdio.h>
#include "font.h"



C12832::C12832(PinName mosi, PinName sck, PinName reset, PinName a0, PinName ncs)
    : _spi(mosi,NC,sck),_reset(reset),_A0(a0),_CS(ncs)
{  
    lcd_reset();
}

void C12832::invert(unsigned int o)
{
    if(o == 0) wr_cmd(0xA6);
    else wr_cmd(0xA7);
}



void C12832::wr_cmd(unsigned char cmd)
{
    _A0 = 0;
    _CS = 0;
    _spi.write(cmd);
    _CS = 1;
}

void C12832::wr_dat(unsigned char dat)
{
    _A0 = 1;
    _CS = 0;
    _spi.write(dat);
    _CS = 1;
}

void C12832::lcd_reset()
{

    _spi.format(8,3);                 // 8 bit spi mode 3
    _spi.frequency(20000000);          // 19,2 Mhz SPI clock
    _A0 = 0;
    _CS = 1;
    _reset = 0;                        // display reset
    wait_us(50);
    _reset = 1;                       // end reset
    wait_ms(5);

    /* Start Initial Sequence ----------------------------------------------------*/

    wr_cmd(0xAE);   //  display off
    wr_cmd(0xA2);   //  bias voltage

    wr_cmd(0xA0);
    wr_cmd(0xC8);   //  colum normal

    wr_cmd(0x22);   //  voltage resistor ratio
    wr_cmd(0x2F);   //  power on
    //wr_cmd(0xA4);   //  LCD display ram
    wr_cmd(0x40);   // start line = 0
    wr_cmd(0xAF);     // display ON

    wr_cmd(0x81);   //  set contrast
    wr_cmd(0x17);   //  set contrast

    wr_cmd(0xA6);     // display normal


    // clear and update LCD
    memset(buffer,0x00,512);  // clear display buffer
    copy_to_lcd();
}

void C12832::pixel(int x, int y, int color)
{
    // first check parameter
    if(x > 127 || y > 31 || x < 0 || y < 0) return;

        if(color == 0)
            buffer[x + ((y/8) * 128)] &= ~(1 << (y%8));  // erase pixel
        else
            buffer[x + ((y/8) * 128)] |= (1 << (y%8));   // set pixel

}

void C12832::copy_to_lcd(void)
{
    
    int i=0;
    
    //page 0
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB0);      // set page address  0
    _A0 = 1;
    for(i=0; i<128; i++) {
        wr_dat(buffer[i]);
    }

    // page 1
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB1);      // set page address  1
    _A0 = 1;
    for(i=128; i<256; i++) {
        wr_dat(buffer[i]);
    }

    //page 2
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB2);      // set page address  2
    _A0 = 1;
    for(i=256; i<384; i++) {
        wr_dat(buffer[i]);
    }

    //page 3
    wr_cmd(0x00);      // set column low nibble 0
    wr_cmd(0x10);      // set column hi  nibble 0
    wr_cmd(0xB3);      // set page address  3
    _A0 = 1;

    _CS = 0;
    
    for(i=384; i<512; i++) {
        wr_dat(buffer[i]);
    }

}

void C12832::cls(void)
{
    memset(buffer,0x00,512);  // clear display buffer
    copy_to_lcd();
}

void C12832::rectangle(int x1, int x2, int y1, int y2)
{
    for(int i=x1;i<=x2;i++)
    {
        pixel(i,y1,1);
        pixel(i,y2,1);
    }
    
    for(int i=y1;i<=y2;i++)
    {
        pixel(x1,i,1);
        pixel(x2,i,1);
    }
}


void C12832::text(int ix, int iy, char* ctext, int length, int size)
{
    int i;
    for(i=0;i<length;i++)
    {
        character(ix+i*6*size,iy,ctext[i],size);    
    }
}


int C12832::character(int ix, int iy, char c, int size)
{
    int code = 0;
    
    if(ix >127 || ix < 0) return -1;
    if(iy >31  || iy < 0) return -1;
    if(c>= 0x20){
        code = (int)(c - 0x20);
    } else if(c == 0x00) {
        code = 96;
    } else {
        return -1;
    }   
    if((size < 0)||(size > 4)) return -1;
    
    int index = code*5;
    unsigned char  codebyte = 0;
    unsigned char codebit = 0;
    
    
    for(int i=0; i< 5 ; i++)
    {
        codebyte = font[index+i];
        for(int j=0; j<7; j++)
        {
             codebit =  codebyte & (1<<j);
             for (int k=0; k<size; k++) {
                for(int l=0;l<size;l++) {
                    pixel(ix+i*size+k,iy+l+j*size,codebit);   
                }             
             }
             
        }     
    } 
    return 0;
}    

void C12832::set_contrast(unsigned char volume)
{
    wr_cmd(0x81);      //  set volume
    wr_cmd(volume & 0x3F);
}   


