#ifndef _HIDSCOPE_H_
#define _HIDSCOPE_H_

#include "mbed.h"
#include "USBHID.h"

/** A simple HID (Human Interface Device) scope
 * - Up to 6 channels of float data is transmitted in a single HID message (64 byte)
 * - Theoretical maximum samplerate of 1kHz
 * - Data can be parsed using a client-side server like NodeJS
 * 
 * Example:
 * @code
 * #include "mbed.h"
 * #include "HIDScope.h"
 *
 * HIDScope    scope(2);
 * Ticker      scopeTimer;
 * 
 * AnalogIn    a0(A0);
 * 
 * int main()
 * {
 *   
 *   scopeTimer.attach_us(&scope, &HIDScope::send, 1e4); // Send data at 100 Hz
 *   
 *   while(1){  // Generate some data
 *       scope.set(0, a0.read());
 *       scope.set(1, a0.read());
 *       wait_us(1000);
 *   };    
 *   
 * }
 * @endcode
*/
class HIDScope {
    public: 
        ///Instantiate the HID Scope
        HIDScope(int channels);
        
        /** Sets the current channel value
          @param ch : integer channel no (0-6)
          @param val : float value
          @return void
        */
        void set(int ch, float val);
        
        /** Sets the current channel value
          @param ch : integer channel no (0-6)
          @param val : integer value
          @return void
        */
        void set(int ch, int val);
        
        /** Sets the current channel value
          @param ch : integer channel no (0-6)
          @param val : boolean value
          @return void
        */
        void set(int ch, bool val);
        
        /** Sets the current channel value
          @param ch : double channel no (0-6)
          @param val : float value
          @return void
        */
        void set(int ch, double val);
        
        /** Sends the channel data to the HID client
          @return void
        */
        void send();
    private:
        USBHID hid;
        HID_REPORT scopeData;
        float* bufferData;
        int channelCount;
};

#endif