/**
 * @file app_task.cpp
 * @brief Handling for ble events and responses.
 * Copyright 2015 SEVENCORE Co., Ltd.
 *
 * @author HyeongJun Kim 
 * @version 1.0.0  
 * @date 2015-08-20
*/
#include "app_task.h"
#include "app.h"
#include "fota_server_task.h"

#define BLE_FOTA_SERVICE 1
/**
 ****************************************************************************************
 * @addtogroup dialog_fota module
 * @brief application handlers for ble events and responses.
 *
 * @{
 ****************************************************************************************
 */
namespace sevencore_fota{
 
/**
 ****************************************************************************************
 * @brief Handles GAPM_ADV_REPORT_IND event.
 *
 * @param[in] msgid     Id of the message received.
 * @param[in] param     Pointer to the parameters of the message.
 * @param[in] dest_id   ID of the receiving task instance (TASK_GAP).
 * @param[in] src_id    ID of the sending task instance.
 *
 * @return If the message was consumed or not.
 ****************************************************************************************
 */   
int gapm_adv_report_ind_handler(unsigned short msgid,
                                struct gapm_adv_report_ind *param,
                                unsigned short dest_id,
                                unsigned short src_id)
{
    //if (app_env.state != APP_SCAN)
    //   return -1;

    return 0;
}
/**
 ****************************************************************************************
 * @brief Handles the DISS_CREATE_DB_CFM message.
 *
 * @param[in] msgid     Id of the message received.
 * @param[in] param     Pointer to the parameters of the message.
 * @param[in] dest_id   ID of the receiving task instance (TASK_GAP).
 * @param[in] src_id    ID of the sending task instance.
 * @param[in] BMH     Ble Message Handler class reference.
 *
 * @return If the message was consumed or not.
 ****************************************************************************************
 */
int diss_create_db_cfm_handler(unsigned short  msgid,
                               struct diss_create_db_cfm *param,
                               unsigned short  dest_id,
                               unsigned short  src_id,
                               BleMsgHandler* BMH)
{
    
    uint8_t len = strlen(APP_DIS_SW_REV);  
      
    if (param->status == CO_ERROR_NO_ERROR)
    {
        char str[22] = "\nDIS SET DB\n";
        BMH->HostPcPrint(str);
        // Set Manufacturer Name value in the DB
        {
            uint8_t *msg;
            struct diss_set_char_val_req req_name;
            unsigned short msg_size = 1+sizeof(ble_hdr)+sizeof(diss_set_char_val_req)-(STR_MAX_LEN-APP_DIS_MANUFACTURER_NAME_STR_LEN);

            // Fill in the parameter structure
            req_name.char_code     = DIS_MANUFACTURER_NAME_CHAR;
            req_name.val_len       = APP_DIS_MANUFACTURER_NAME_STR_LEN;
            memcpy(&req_name.val[0], APP_DIS_MANUFACTURER_NAME_STR, APP_DIS_MANUFACTURER_NAME_STR_LEN);

            msg = new uint8_t[msg_size];
            
            BMH->BleMsgAlloc(DISS_SET_CHAR_VAL_REQ,TASK_DISS,TASK_GTL,
                sizeof(struct diss_set_char_val_req)-(STR_MAX_LEN-APP_DIS_MANUFACTURER_NAME_STR_LEN),&req_name,msg);
            BMH->BleSendMsg(msg,msg_size);
            free(msg);
            
        }

        // Set Model Number String value in the DB
        {
            uint8_t *msg;
            struct diss_set_char_val_req req_mod;
            unsigned short msg_size = 1+sizeof(ble_hdr)+sizeof(diss_set_char_val_req)-(STR_MAX_LEN-APP_DIS_MODEL_NB_STR_LEN);

            // Fill in the parameter structure
            req_mod.char_code     = DIS_MODEL_NB_STR_CHAR;
            req_mod.val_len       = APP_DIS_MODEL_NB_STR_LEN;
            memcpy(&req_mod.val[0], APP_DIS_MODEL_NB_STR, APP_DIS_MODEL_NB_STR_LEN);

            msg = new uint8_t[msg_size];
            
            BMH->BleMsgAlloc(DISS_SET_CHAR_VAL_REQ,TASK_DISS, TASK_GTL,
                    sizeof(struct diss_set_char_val_req)-(STR_MAX_LEN-APP_DIS_MODEL_NB_STR_LEN),&req_mod,msg);
            // Send the message
            BMH->BleSendMsg(msg,msg_size);
            free(msg);
        }
#if (BLE_FOTA_SERVICE)
        // Set Serial Number String value in the DB
        {
            uint8_t *msg;
            struct diss_set_char_val_req req_ser;
            unsigned short msg_size = 1+sizeof(ble_hdr)+sizeof(diss_set_char_val_req)-(STR_MAX_LEN-APP_DIS_SERIAL_NB_STR_LEN);

            // Fill in the parameter structure
            req_ser.char_code     = DIS_SERIAL_NB_STR_CHAR;
            req_ser.val_len       = APP_DIS_SERIAL_NB_STR_LEN;
            memcpy(&req_ser.val[0], APP_DIS_SERIAL_NB_STR, APP_DIS_SERIAL_NB_STR_LEN);

            msg = new uint8_t[msg_size];
            
            BMH->BleMsgAlloc(DISS_SET_CHAR_VAL_REQ,TASK_DISS, TASK_GTL,
                    sizeof(struct diss_set_char_val_req)-(STR_MAX_LEN-APP_DIS_SERIAL_NB_STR_LEN),&req_ser,msg);
            // Send the message
            BMH->BleSendMsg(msg,msg_size);
            free(msg);
        }
#endif //(BLE_FOTA_SERVICE)
#if !(BLE_FOTA_SERVICE)       
        // Set System ID value in the DB
        {
            uint8_t *msg;
            struct diss_set_char_val_req req_id;
            unsigned short msg_size = 1+sizeof(ble_hdr)+sizeof(diss_set_char_val_req);

            // Fill in the parameter structure
            req_id.char_code     = DIS_SYSTEM_ID_CHAR;
            req_id.val_len       = APP_DIS_SYSTEM_ID_LEN;
            memcpy(&req_id.val[0], APP_DIS_SYSTEM_ID, APP_DIS_SYSTEM_ID_LEN);
            
            msg = new uint8_t[msg_size];
            
            BMH->BleMsgAlloc(DISS_SET_CHAR_VAL_REQ,TASK_DISS, TASK_GTL,sizeof(struct diss_set_char_val_req),&req_id,msg);

            // Send the message
            BMH->BleSendMsg(msg, msg_size);
            free(msg);
        }            
        

        // Set the software version in the DB
        {
            uint8_t *msg;
            struct diss_set_char_val_req req_id;
            unsigned short msg_size = 1+sizeof(ble_hdr)+sizeof(diss_set_char_val_req);
            
            // Fill in the parameter structure
            req_id.char_code     = DIS_SW_REV_STR_CHAR;
            req_id.val_len       = len;
            memcpy(&req_id.val[0], APP_DIS_SW_REV, len);
            
            msg = new uint8_t[msg_size];

            BMH->BleMsgAlloc(DISS_SET_CHAR_VAL_REQ,TASK_DISS, TASK_GTL,sizeof(struct diss_set_char_val_req),&req_id,msg);
            // Send the message
            BMH->BleSendMsg(msg,msg_size);
            free(msg);
        }

        len = strlen(APP_DIS_FIRM_REV);
        // Set the firmware version in the DB. This is the common code sw version
        {
            uint8_t *msg;
            struct diss_set_char_val_req req_id;
            unsigned short msg_size = 1+sizeof(ble_hdr)+sizeof(diss_set_char_val_req);
            
            // Fill in the parameter structure
            req_id.char_code     = DIS_FIRM_REV_STR_CHAR;
            req_id.val_len       = len;
            memcpy(&req_id.val[0], APP_DIS_FIRM_REV, len);
            
            msg = new uint8_t[msg_size];

            BMH->BleMsgAlloc(DISS_SET_CHAR_VAL_REQ,TASK_DISS, TASK_GTL,sizeof(struct diss_set_char_val_req),&req_id,msg);
            // Send the message
            BMH->BleSendMsg(msg, msg_size);
            free(msg);
        }
#endif //!(BLE_FOTA_SERVICE)
    }
    
    if (app_env.state == APP_IDLE)
    {
        char str[30] = "\nfota db create req!!\n";
        BMH->HostPcPrint(str);
        //app_set_mode(BMH);
        app_fota_server_db_create(BMH);
    }
        
    return 0;
}
/**
 ****************************************************************************************
 * @brief Handles the FOTA_SERVER_CREATE_DB_CFM message.
 *
 * @param[in] msgid     Id of the message received.
 * @param[in] param     Pointer to the parameters of the message.
 * @param[in] dest_id   ID of the receiving task instance (TASK_GAP).
 * @param[in] src_id    ID of the sending task instance.
 * @param[in] BMH     Ble Message Handler class reference.
 *
 * @return If the message was consumed or not.
 ****************************************************************************************
 */
int fota_server_create_db_cfm_handler(unsigned short  msgid,
                               struct fota_server_create_db_cfm *param,
                               unsigned short  dest_id,
                               unsigned short  src_id,
                               BleMsgHandler* BMH)
{ 
    uint8_t fota_normal = 0;
    uint8_t fota_normal_len = sizeof(fota_normal);
    if (param->status == CO_ERROR_NO_ERROR)
    {
        // Initialization Firmware new Version Char ("00-00-00")
        {
            uint8_t *msg;
            struct fota_server_set_char_val_req req_nv;
            unsigned short msg_size = 1+sizeof(ble_hdr)+ sizeof(fota_server_set_char_val_req)-(18-APP_FOTA_SERVER_FIRMWARE_NEW_VERSION_LEN);
            // Fill in the parameter structure
            req_nv.char_code     = FOTA_SERVER_FIRMWARE_NEW_VERSION_CHAR;
            req_nv.val_len       = APP_FOTA_SERVER_FIRMWARE_NEW_VERSION_LEN;
            memcpy(&req_nv.val[0], APP_FOTA_SERVER_FIRMWARE_NEW_VERSION, APP_FOTA_SERVER_FIRMWARE_NEW_VERSION_LEN);
            
            msg = new uint8_t[msg_size];
            
            BMH->BleMsgAlloc(FOTA_SERVER_SET_CHAR_VAL_REQ,TASK_FOTA_SERVER, TASK_GTL,sizeof(struct fota_server_set_char_val_req)-(18-APP_FOTA_SERVER_FIRMWARE_NEW_VERSION_LEN),&req_nv,msg );
            // Send the message
            BMH->BleSendMsg(msg,msg_size);
            free(msg);
        }

        // Initialization Sequence Number Char ( = '0' )
        {
            uint8_t *msg;
            struct fota_server_set_char_val_req req_sn;
            unsigned short msg_size = 1+sizeof(ble_hdr)+sizeof(fota_server_set_char_val_req)-(18-fota_normal_len);

            // Fill in the parameter structure
            req_sn.char_code     = FOTA_SERVER_SEQUENCE_NUMBER_CHAR;
            req_sn.val_len       = fota_normal_len;
            memcpy(&req_sn.val[0], &fota_normal, fota_normal_len);

            msg = new uint8_t[msg_size];
            
            BMH->BleMsgAlloc(FOTA_SERVER_SET_CHAR_VAL_REQ,TASK_FOTA_SERVER, TASK_GTL,sizeof(struct fota_server_set_char_val_req)-(18-fota_normal_len),&req_sn,msg );
            // Send the message
            BMH->BleSendMsg(msg,msg_size);
            free(msg);
        }
        
        // Initialization Firmware Data Check Char ( = '0' )
        {
            uint8_t *msg;
            struct fota_server_set_char_val_req req_fdc;
            unsigned short msg_size = 1+sizeof(ble_hdr)+sizeof(fota_server_set_char_val_req)-(18-fota_normal_len);

            // Fill in the parameter structure
            req_fdc.char_code     = FOTA_SERVER_FIRMWARE_DATA_CHECK_CHAR;
            req_fdc.val_len       = fota_normal_len;
            memcpy(&req_fdc.val[0], &fota_normal, fota_normal_len);

            msg = new uint8_t[msg_size];
            
            BMH->BleMsgAlloc(FOTA_SERVER_SET_CHAR_VAL_REQ,TASK_FOTA_SERVER, TASK_GTL,sizeof(struct fota_server_set_char_val_req)-(18-fota_normal_len),&req_fdc,msg );
            // Send the message
            BMH->BleSendMsg(msg,msg_size);
            free(msg);
        }
        
        // Initialization Firmware Status Char ( = '0' )
        {
            uint8_t *msg;
            struct fota_server_set_char_val_req req_stat;
            unsigned short msg_size = 1+sizeof(ble_hdr)+sizeof(fota_server_set_char_val_req)-(18-fota_normal_len);

            // Fill in the parameter structure
            req_stat.char_code     = FOTA_SERVER_FIRMWARE_STATUS_CHAR;
            req_stat.val_len       = fota_normal_len;
            memcpy(&req_stat.val[0],  &fota_normal, fota_normal_len);

            msg = new uint8_t[msg_size];
            
            BMH->BleMsgAlloc(FOTA_SERVER_SET_CHAR_VAL_REQ,TASK_FOTA_SERVER, TASK_GTL,sizeof(struct fota_server_set_char_val_req)-(18-fota_normal_len),&req_stat,msg );
            // Send the message
            BMH->BleSendMsg(msg,msg_size);
            free(msg);
        }
        
        // Initialization Reset Char ( = '0' )
        {
            uint8_t *msg;
            struct fota_server_set_char_val_req req_reset;
            unsigned short msg_size = 1+sizeof(ble_hdr)+sizeof(fota_server_set_char_val_req)-(18-fota_normal_len);

            // Fill in the parameter structure
            req_reset.char_code     = FOTA_SERVER_RESET_CHAR;
            req_reset.val_len       = fota_normal_len;
            memcpy(&req_reset.val[0], &fota_normal, fota_normal_len);

            msg = new uint8_t[msg_size];
            
            BMH->BleMsgAlloc(FOTA_SERVER_SET_CHAR_VAL_REQ,TASK_FOTA_SERVER, TASK_GTL,sizeof(struct fota_server_set_char_val_req)-(18-fota_normal_len),&req_reset,msg );
            // Send the message
            BMH->BleSendMsg(msg,msg_size);
            free(msg);
        }
        
    }
    
    if (app_env.state == APP_IDLE)
    {
        app_set_mode(BMH);
    }
    
    return 0;
}
/**
 ****************************************************************************************
 * @brief Handles Firmware Data Copy event.
 *
 * @param[in] msgid     Id of the message received.
 * @param[in] param     Pointer to the parameters of the message.
 * @param[in] dest_id   ID of the receiving task instance (TASK_GAP).
 * @param[in] src_id    ID of the sending task instance.
 * @param[in] BMH     Ble Message Handler class reference.
 *
 * @return If the message was consumed or not.
 ****************************************************************************************
 */
void fota_server_data_flash_ind_handler(unsigned short msgid,
                                  struct fota_server_data_flash_ind *param,
                                  unsigned short dest_id,
                                  unsigned short src_id,
                                  BleMsgHandler* BMH)
{
    char version[9];
    memcpy(version,param->version,8);
    version[8] = '\0';
    BMH->FirmwareDataReceive(param->code_size,version);
}
/**
 ****************************************************************************************
 * @brief Handles ready indication from the GAP.
 *
 * @param[in] msgid     Id of the message received.
 * @param[in] param     Pointer to the parameters of the message.
 * @param[in] dest_id   ID of the receiving task instance (TASK_GAP).
 * @param[in] src_id    ID of the sending task instance.
 * @param[in] BMH     Ble Message Handler class reference.
 *
 * @return If the message was consumed or not.
 ****************************************************************************************
 */
int gapm_device_ready_ind_handler(unsigned short msgid,
                                  struct gap_ready_evt *param,
                                  unsigned short dest_id,
                                  unsigned short src_id,
                                  BleMsgHandler* BMH)
{
    // We are now in Connectable State
    if (dest_id == TASK_GTL)
    {
        app_rst_gap(BMH);
    }
    
    return 0;
}
/**
 ****************************************************************************************
 * @brief Handle reset GAP request completion event.
 *
 * @param[in] msgid     Id of the message received.
 * @param[in] param     Pointer to the parameters of the message.
 * @param[in] dest_id   ID of the receiving task instance (TASK_GAP).
 * @param[in] src_id    ID of the sending task instance.
 * @param[in] BMH     Ble Message Handler class reference.
 *
 * @return If the message was consumed or not.
 ****************************************************************************************
 */ 
int gapm_reset_completion_handler(unsigned short msgid,
                                  struct gapm_cmp_evt *param,
                                  unsigned short dest_id,
                                  unsigned short src_id,
                                  BleMsgHandler* BMH)
{
    // We are now in Connectable State
    if (dest_id == TASK_GTL)
    {
        app_env.state = APP_IDLE;           
        app_diss_db_create(BMH);
    }
    
    return 0;
}
/**
 ****************************************************************************************
 * @brief Handles GAPM_CMP_EVT event for GAPM_SET_DEV_CONFIG_CMD.
 *
 * @param[in] msgid     Id of the message received.
 * @param[in] param     Pointer to the parameters of the message.
 * @param[in] dest_id   ID of the receiving task instance (TASK_GAP).
 * @param[in] src_id    ID of the sending task instance.
 * @param[in] BMH     Ble Message Handler class reference.
 *
 * @return If the message was consumed or not.
 ****************************************************************************************
 */
int gapm_set_dev_config_completion_handler(unsigned short msgid,
                                           struct gapm_cmp_evt *param,
                                           unsigned short dest_id,
                                           unsigned short src_id,
                                           BleMsgHandler* BMH)
{
    app_env.state = APP_CONNECTABLE;
    wait(1);
    app_adv_start(BMH); // start advertising

    return 0;
}
/**
 ****************************************************************************************
 * @brief Handles Connection request indication event.
 *
 * @param[in] msgid     Id of the message received.
 * @param[in] param     Pointer to the parameters of the message.
 * @param[in] dest_id   ID of the receiving task instance (TASK_GAP).
 * @param[in] src_id    ID of the sending task instance.
 * @param[in] BMH     Ble Message Handler class reference.
 *
 * @return If the message was consumed or not.
 ****************************************************************************************
 */
int gapc_connection_req_ind_handler(uint16_t msgid,
                                    struct gapc_connection_req_ind *param,
                                    uint16_t dest_id,
                                    uint16_t src_id,
                                    BleMsgHandler* BMH)
{
    app_env.state = APP_CONNECTED;
    
    ble_dev device;
    
    // Retrieve the connection index from the GAPC task instance for the connection
    device.conidx = KE_IDX_GET(src_id);
        
    // Retrieve the connection handle from the parameters
    device.conhdl = param->conhdl;

    memcpy(device.adv_addr.addr, param->peer_addr.addr, sizeof(struct bd_addr));
     
    
    char str[30] = "\nDIS&FOTA ENABLE and CONFIRM\n";
    BMH->HostPcPrint(str);
    
    app_dis_enable(&device,BMH);
    app_fota_server_enable(&device,BMH);
    app_connect_confirm(GAP_AUTH_REQ_NO_MITM_NO_BOND,&device,BMH);
   
    return 0;
}
/**
 ****************************************************************************************
 * @brief Handles Discconnection indication event.
 *
 * @param[in] msgid     Id of the message received.
 * @param[in] param     Pointer to the parameters of the message.
 * @param[in] dest_id   ID of the receiving task instance (TASK_GAP).
 * @param[in] src_id    ID of the sending task instance.
 * @param[in] BMH     Ble Message Handler class reference.
 *
 * @return If the message was consumed or not.
 ****************************************************************************************
 */
int gapc_disconnect_ind_handler(uint16_t msgid,
                                struct gapc_disconnect_ind *param,
                                uint16_t dest_id,
                                uint16_t src_id,
                                BleMsgHandler* BMH)
{
    char str[22] = "\nDevice Disconnected\n";
    BMH->HostPcPrint(str);
    
    app_send_disconnect(TASK_FOTA_SERVER, param->conhdl, param->reason, BMH);  
    app_env.state = APP_IDLE;
    
    wait(1);
    app_set_mode(BMH);
    
    return 0;
}


}//namespace

/// @} dialog_fota module
