#include <stdio.h>
#include "BME280.h"

#include "lorawan/LoRaWANInterface.h"
#include "lorawan/system/lorawan_data_structures.h"
#include "events/EventQueue.h"

#include "lora_radio_helper.h"

using namespace events;

/*
 * Sets up an application dependent transmission timer in ms. Used only when Duty Cycling is off for testing
 */
#define TX_TIMER                        10000

/**
 * Maximum number of events for the event queue.
 * 10 is the safe number for the stack events, however, if application
 * also uses the queue for whatever purposes, this number should be increased.
 */
#define MAX_NUMBER_OF_EVENTS            10

/**
 * Maximum number of retries for CONFIRMED messages before giving up
 */
#define CONFIRMED_MSG_RETRY_COUNTER     3

#define DEBUG   1
#define BUFFER_SIZE     8


// Max payload size can be LORAMAC_PHY_MAXPAYLOAD.
// Use set_max_payload_length() function to change maximum payload capacity.

uint8_t tx_buffer[BUFFER_SIZE];

BME280 sensor(PB_9, PB_8, 0x77);

uint16_t packet_len = BUFFER_SIZE;
float sensor_values[2];
    

/**
* This event queue is the global event queue for both the
* application and stack. To conserve memory, the stack is designed to run
* in the same thread as the application and the application is responsible for
* providing an event queue to the stack that will be used for ISR deferment as
* well as application information event queuing.
*/

static EventQueue ev_queue(MAX_NUMBER_OF_EVENTS *EVENTS_EVENT_SIZE);

/**
 * Event handler.
 *
 * This will be passed to the LoRaWAN stack to queue events for the
 * application which in turn drive the application.
 */
static void lora_event_handler(lorawan_event_t event);

/**
 * Constructing Mbed LoRaWANInterface and passing it the radio object from lora_radio_helper.
 */
static LoRaWANInterface lorawan(radio);

/**
 * Application specific callbacks
 */
static lorawan_app_callbacks_t callbacks;
static void send_message();


int main(void)
{

    // stores the status of a call to LoRaWAN protocol
    lorawan_status_t retcode;

    // Initialize LoRaWAN stack
    if (lorawan.initialize(&ev_queue) != LORAWAN_STATUS_OK) {
        if(DEBUG)
            printf("\r\n LoRa initialization failed! \r\n");
        return -1;
    }

    if(DEBUG)
        printf("\r\n Mbed LoRaWANStack initialized \r\n");

    // prepare application callbacks
    callbacks.events = mbed::callback(lora_event_handler);
    lorawan.add_app_callbacks(&callbacks);

    // Set number of retries in case of CONFIRMED messages
    if (lorawan.set_confirmed_msg_retries(CONFIRMED_MSG_RETRY_COUNTER)
            != LORAWAN_STATUS_OK) {
        if(DEBUG)
            printf("\r\n set_confirmed_msg_retries failed! \r\n\r\n");
        return -1;
    }

    if(DEBUG)
        printf("\r\n CONFIRMED message retries : %d \r\n", CONFIRMED_MSG_RETRY_COUNTER);

    // Enable adaptive data rate
    if (lorawan.enable_adaptive_datarate() != LORAWAN_STATUS_OK) {
        if(DEBUG)
            printf("\r\n enable_adaptive_datarate failed! \r\n");
        return -1;
    }

    if(DEBUG)
        printf("\r\n Adaptive data  rate (ADR) - Enabled \r\n");

    retcode = lorawan.connect();

    if (retcode == LORAWAN_STATUS_OK || retcode == LORAWAN_STATUS_CONNECT_IN_PROGRESS) {
    } else {
        if(DEBUG)
            printf("\r\n Connection error, code = %d \r\n", retcode);
        return -1;
    }

    if(DEBUG)
        printf("\r\n Connection - In Progress ...\r\n");

    
    // make your event queue dispatching events forever
    ev_queue.dispatch_forever();
    
    return 0;
}

/**
 * Sends a message to the Network Server
 */
static void send_message()
{
    int16_t retcode;
    
    sensor.trigger();
    sensor_values[0] = sensor.getTemperature();
    sensor_values[1] = sensor.getPressure();
    //sensor_values[2] = sensor.getHumidity();

    memcpy(tx_buffer, sensor_values, packet_len);
    
    retcode = lorawan.send(MBED_CONF_LORA_APP_PORT, tx_buffer, packet_len, MSG_UNCONFIRMED_FLAG);

    if (retcode < 0) {
        
        if(DEBUG)
            retcode == LORAWAN_STATUS_WOULD_BLOCK ? printf("send - WOULD BLOCK\r\n")
                : printf("\r\n send() - Error code %d \r\n", retcode);

        if (retcode == LORAWAN_STATUS_WOULD_BLOCK) {
            //retry in 3 seconds
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                ev_queue.call_in(3000, send_message);
            }
        }
        return;
    }

    if(DEBUG)
        printf("\r\n Temp: %f \r\n", sensor_values[0]);
        printf("\r\n Pressure: %f \r\n", sensor_values[1]);
        printf("\r\n txBuffer: %.2X %.2X %.2X %.2X %.2X %.2X %.2X %.2X \r\n", tx_buffer[0], tx_buffer[1], tx_buffer[2], tx_buffer[3], tx_buffer[4], tx_buffer[5], tx_buffer[6], tx_buffer[7]);
        printf("\r\n %d bytes scheduled for transmission \r\n", retcode);
    
    //Check if radio wakes up automatically. If not, then call radio.TX();
    radio.sleep();
}

/**
 * Receive a message from the Network Server
 */
static void receive_message()
{
    
}

/**
 * Event handler
 */
static void lora_event_handler(lorawan_event_t event)
{
    switch (event) {
        case CONNECTED:
            if(DEBUG)
                printf("\r\n Connection - Successful \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            } else {
                ev_queue.call_every(TX_TIMER, send_message);
            }

            break;
        case DISCONNECTED:
            ev_queue.break_dispatch();
            if(DEBUG)
                printf("\r\n Disconnected Successfully \r\n");
            break;
        case TX_DONE:
            if(DEBUG)
                printf("\r\n Message Sent to Network Server \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            }
            break;
        case TX_TIMEOUT:
        case TX_ERROR:
        case TX_CRYPTO_ERROR:
        case TX_SCHEDULING_ERROR:
            if(DEBUG)
                printf("\r\n Transmission Error - EventCode = %d \r\n", event);
            // try again
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            }
            break;
        case RX_DONE:
            if(DEBUG)
                printf("\r\n Received message from Network Server \r\n");
            receive_message();
            break;
        case RX_TIMEOUT:
        case RX_ERROR:
            if(DEBUG)
                printf("\r\n Error in reception - Code = %d \r\n", event);
            break;
        case JOIN_FAILURE:
            if(DEBUG)
                printf("\r\n OTAA Failed - Check Keys \r\n");
            break;
        case UPLINK_REQUIRED:
            if(DEBUG)
                printf("\r\n Uplink required by NS \r\n");
            if (MBED_CONF_LORA_DUTY_CYCLE_ON) {
                send_message();
            }
            break;
        default:
            MBED_ASSERT("Unknown Event");
    }
}

// EOF
