/*
 * SerialServer.cpp
 * Copyright (c) 2017, ZHAW
 * All rights reserved.
 *
 *  Created on: 05.06.2017
 *      Author: Marcel Honegger
 */


#include <mbed.h>
#include <iostream>
#include <string>
#include <sstream>
#include <vector>
#include "SerialCom.h"

using namespace std;

const float SerialCom::PERIOD = 0.0001f;     // the period of the timer interrupt, given in [s]
const char SerialCom::DELIM = ',';

/**
 * Create a serial server object.
 

 
SerialServer::SerialServer(RawSerial& serial, StateMachine& stateMachine) : serial(serial), stateMachine(stateMachine), thread(osPriorityRealtime, STACK_SIZE) {
*/
SerialCom::SerialCom(RawSerial& serial) : serial(serial), thread(osPriorityRealtime, STACK_SIZE) { 
    // initialize local values
    
    input.clear();
    
    // start thread and ticker timer interrupt service routine
    
    thread.start(callback(this, &SerialCom::run));
    ticker.attach(callback(this, &SerialCom::sendSignal), PERIOD);
}

/**
 * Delete the serial server object and release all allocated resources.
 */
SerialCom::~SerialCom() {
    
    ticker.detach();
}

/**
 * This method is called by the ticker timer interrupt service routine.
 * It sends a signal to the thread to make it run again.
 */
void SerialCom::sendSignal() {
    
    thread.signal_set(signal);
}

/**
 * This <code>run()</code> method contains an infinite loop with the run logic.
 */
void SerialCom::run() {
    
    Serial pc(PA_2, PA_3); // tx, rx
    pc.baud(9600);
    
    while (true) {
        
        // wait for the periodic signal
        
        thread.signal_wait(signal);
        
        
        
        // read received characters while buffer is full
        
        while (serial.readable()) {
            int32_t c = serial.getc();
            //if (input.size() >= MESSAGE_SIZE) input.erase(0, 1);
            input += static_cast<uint8_t>(c);
        }
        
        // try to decode a message
        
        if (input[input.size()-1] != '&') {
            
            // message is too short, keep reading...
            
        } else {
            pc.printf("command: %s", input);
            splitString();
            if (tokens[0] == "M") {
                if (tokens.size() == 45+2) {
                    float newK[3][15];
                    int y = 0;
                    int x = 0;
                    float checksum = 0;       
                    for (int i=1; i < tokens.size()-1; i++) {
                        newK[y][x] = strtof((tokens[i]).c_str(), 0);
                        pc.printf("Element %d ,  %d: %.10f\r\n", y, x, newK[y][x]);
                        x++;
                        checksum += newK[y][x];
                        if (x == 15){
                            x = 0;
                            y++;
                        }
                    }
                }
                else{
                    pc.printf("Data to short od long\r\n");
                }
                  
            }
            else if (tokens[0] == "G") {
                pc.printf("Send data to pc\r\n");
                if(sendData == false) {
                    sendData = true;
                    runCount = 0;
                }
                else{
                    sendData = false;
                }
                
            }
            else{
                pc.printf("Command not understood\r\n");    
            }
            
           
            tokens.clear();
            input.clear();

      
        }
        if(sendData == true) {
            if (output.size() == 0 && runCount >= 1000){
                output += "Hello nicolas\n";
                runCount = 0;
            }
        } 
        runCount++;
        while (serial.writeable() && (output.size() > 0)) {
            serial.putc(output[0]);
            output.erase(0, 1);
        }    

    }
    
}


void SerialCom::splitString() {
    stringstream ss(input);
    string item;
    while (getline(ss, item, DELIM)) {
        tokens.push_back(item);
   } 
    return;
}
