#include "main.h"

Serial pc(USBTX, USBRX);
Serial x1(p28, p27);         // RS485 auf X1
Serial x2(p13, p14);         // RS232 auf X2
Serial x3(p9, p10);          // RS232 auf X3

DigitalOut MAIN_LED(LED1);
DigitalOut X1_LED(LED2);
DigitalOut X2_LED(LED3);
DigitalOut X3_LED(LED4);
DigitalOut X1_RX_EN(p26);    // Enable RX on X1
DigitalOut LED_LNK(p11);     // Link LED Ethernet
DigitalOut LED_SPD(p12);     // Active LED Ethernet
DigitalOut OUT1(p21);        // Output 1
DigitalOut OUT2(p22);        // Output 2
DigitalOut OUT3(p23);        // Output 3
DigitalOut OUT4(p24);        // Output 4

DigitalIn  IN1 (p15, PullUp);  // Input 1
DigitalIn  IN2 (p16, PullUp);  // Input 2
DigitalIn  IN3 (p17, PullUp);  // Input 3
DigitalIn  IN4 (p18, PullUp);  // Input 4

DigitalIn  LED_LNK_ORG (P1_25, PullUp);    // Die internen Anschluesse der Ethernet LEDs
DigitalIn  LED_SPD_ORG (P1_26, PullUp);

AnalogIn   UMESS(p19);       // Analog Spannung 36.3V

CAN  can(p30, p29, BAUD_CAN); // CAN Schnittstelle

Ticker     toggle_tmr;
Timeout    led1_tmr, led2_tmr, led3_tmr;

static char    x1_in[260], x2_in[260], x3_in[260], x1_buf[260], x2_buf[260], x3_buf[260];
static char    x1_out[260];
static uint8_t x1_cnt, x1_new, x2_cnt, x2_new, x3_cnt, x3_new, x1_len, x2_len, x3_len;
static uint8_t in, in_old;

// --------------------------------------------------------------------------------------

// Diese Funktionen werden vom Timer aufgerufen, LED nach x msec ausschalten

void x1_led_off (void) { X1_LED = 0; }
void x2_led_off (void) { X2_LED = 0; }
void x3_led_off (void) { X3_LED = 0; }

// --------------------------------------------------------------------------------------

// Datenausgabe an RS485 kann nicht direkt erfolgen, da die Empfang
// für die Dauer des Sendens deaktiviert werden muss. Sonst wuerde die
// gesendete Botschaft wieder zurueckgespiegelt werden.

void x1_printf (char * ptr)
{
//  va_list ap;
  
  X1_RX_EN = 1;       // Bei RS485 muss beim Senden der Empfang deaktiviert werden
  //va_start(ap, ptr);  // va_start/va_end sind fuer die Uebergabe der printf Paramter notwendig
  x1.printf(ptr);//, ap);
  //va_end(ap);
  wait(0.001);
  while (!(LPC_UART2->LSR & (1 << 6)));  // Warten auf Ende der Uebertragung (leider direkt in HW notwendig)
  wait(0.005);        // noch 5ms extra warten
  X1_RX_EN = 0;       // Empfang wieder aktivieren
}

// --------------------------------------------------------------------------------------

// Wird vom UART Interrupt bei Empfang eines Zeichens aufgerufen, hier: RS485
void x1_callback (void)
{
  uint8_t ch;
  X1_LED = 1;
  led1_tmr.attach (&x1_led_off, LED_AKTIV_ZEIT);  // LED nach 0.1sec wieder aus
  
  ch = x1.getc();
  if (ch != 0x0A)
  {
    x1_in[x1_cnt++] = ch;
    if (ch == 0x0D)
    {
      x1_in[x1_cnt+1] = 0x00;
      memcpy(x1_buf, x1_in, x1_cnt+1);
      x1_len = x1_cnt;
      x1_cnt = 0;
      x1_new = 1;
    } 
  }
}

// --------------------------------------------------------------------------------------

// Wird vom UART Interrupt bei Empfang eines Zeichens aufgerufen, hier: RS232 an X2
void x2_callback (void)
{
  uint8_t ch;
  X2_LED = 1;
  led2_tmr.attach (&x2_led_off, LED_AKTIV_ZEIT);  // LED nach 0.1sec wieder aus per Timer

  ch = x2.getc();
  if (ch != 0x0A)                      // 0x0A LF wird komplett ignoriert
  {
    x2_in[x2_cnt++] = ch;              // In Empfangspuffer eintragen
    if (ch == 0x0D)                    // Sobald 0x0D CR empfangen
    {
      x2_in[x2_cnt+1] = 0x00;          // Mit 0x00 abschließen
      memcpy(x2_buf, x2_in, x2_cnt+1); // In 2. Buffer kopieren
      x2_len = x2_cnt;                 // Laenge uebergeben
      x2_cnt = 0;                      // Empfangspuffer ruecksetzen
      x2_new = 1;                      // Flag fuer neue Nachricht setzen
    } 
  }
}

// --------------------------------------------------------------------------------------

// Wird vom UART Interrupt bei Empfang eines Zeichens aufgerufen, hier: RS232 an X3
void x3_callback (void)
{
  uint8_t ch;

  X3_LED = 1;
  led3_tmr.attach (&x3_led_off, LED_AKTIV_ZEIT);  // Kommentare s.o.

  ch = x3.getc();
  if (ch != 0x0A)
  {
    x3_in[x3_cnt++] = ch;
    if (ch == 0x0D)
    {
      x3_in[x3_cnt+1] = 0x00;
      memcpy(x3_buf, x3_in, x3_cnt+1);
      x3_len = x3_cnt;
      x3_cnt = 0;
      x3_new = 1;
    } 
  }
}

// --------------------------------------------------------------------------------------

// PC_Callback fuer Empfang von USB-Schnittstelle (nur für Debug)
// Wird wie RS485 Buffer gehandhabt
void pc_callback (void)
{
  uint8_t ch;

  ch = pc.getc();
  if (ch != 0x0A)
  {
    x1_in[x1_cnt++] = ch;
    if (ch == 0x0D)
    {
      x1_in[x1_cnt+1] = 0x00;
      memcpy(x1_buf, x1_in, x1_cnt+1);
      x1_len = x1_cnt;
      x1_cnt = 0;
      x1_new = 1;
    } 
  }
}

// --------------------------------------------------------------------------------------

void check_inputs (uint8_t force)
{
  char buf[16] = "INP:0000\r\0";                // Antwortstring vorbelegen
  
  if (!IN1) {in |= 0x01; } else { in &= ~0x01; }  // Aus HW lesen und bitweise "in" zusammensetzen (Bit 0..3)
  if (!IN2) {in |= 0x02; } else { in &= ~0x02; }
  if (!IN3) {in |= 0x04; } else { in &= ~0x04; }
  if (!IN4) {in |= 0x08; } else { in &= ~0x08; }
  
  if ((in != in_old) || force)         // Wert veraendert oder Ausgabe erzwingen?
  {
    in_old = in;                       // Wert fuer naechsten Vergleich merken
    if (in & 1) buf[4] = '1';
    if (in & 2) buf[5] = '1';          // Ausgabestring manipulieren
    if (in & 4) buf[6] = '1';
    if (in & 8) buf[7] = '1';
    x1_printf(buf);                    // String ausgeben
  }
}

// --------------------------------------------------------------------------------------

void set_outputs (void)
{
  if (x1_buf[4] == '1') OUT1=0; else OUT1=1;  // invertierte Ausgangslogik
  if (x1_buf[5] == '1') OUT2=0; else OUT2=1;
  if (x1_buf[6] == '1') OUT3=0; else OUT3=1;
  if (x1_buf[7] == '1') OUT4=0; else OUT4=1;
}

// --------------------------------------------------------------------------------------

void led_blink (void)
{
  MAIN_LED = !MAIN_LED;               // MAIN_LED ist Blinkeled zur Lebensmeldung
}

// --------------------------------------------------------------------------------------

void copy_eth_leds (void)
{
  LED_LNK = LED_LNK_ORG;
  LED_SPD = LED_SPD_ORG;
}

// --------------------------------------------------------------------------------------

int main() 
{
  x1.baud(BAUD_X1_RS485);
  x2.baud(BAUD_X2_RS232);
  x3.baud(BAUD_X3_RS232);

  toggle_tmr.attach(&led_blink, LED_BLINK_ZEIT);
  
  OUT1 = !OUT1_DEF; 
  OUT2 = !OUT2_DEF; 
  OUT3 = !OUT3_DEF; 
  OUT4 = !OUT4_DEF;
  X1_RX_EN = 0;                        // Empfang RS485 freischalten
  x1.attach(&x1_callback);             // UART-Callbacks aktivieren
  x2.attach(&x2_callback);
  x3.attach(&x3_callback);
  pc.attach(&pc_callback);
  wait(0.01);                          // Nach Umschaltung RS485 etwas warten
  x1_cnt = 0; x2_cnt = 0; x3_cnt = 0;  // Alle Emfangsbuffer ruecksetzen
  
  while(1)                             // Hauptschleife
  {
    if (x2_new)                        // Neuer Eingabestring von Port X2
    {
      x1_out[0] = '2';                 // "2:" voranstellen
      x1_out[1] = ':';
      memcpy(&x1_out[2], &x2_buf[0], x2_len+1);  // Nachricht kopieren
      x1_printf(x1_out);               // und an RS485 ausgeben
      x2_new = 0;
    }    

    if (x3_new)                        // Neuer Eingabestring von X3?
    {
      x1_out[0] = '3';                 // "3:" voranstellen
      x1_out[1] = ':';
      memcpy(&x1_out[2], &x3_buf[0], x3_len+1);
      x1_printf(x1_out);               // RS485 Ausgabe
      x3_new = 0;
    } 
    
    if (x1_new)                        // Neuer Eingabestring von X1 (RS485)
    {
      if ((!strncasecmp(x1_buf, "OUT:", 4)) && (x1_len >= 8))
      {
        set_outputs();                 // Ausgaenge stzen
        x1_buf[0] = 'S'; 
        x1_buf[1] = 'E';
        x1_buf[2] = 'T';
        x1_printf(x1_buf);             // Zur Bestaetigung wiederholen "SET:xxxx"
      }
      else if ((!strncasecmp(x1_buf, "INP?", 4)) && (x1_len >= 4))
      {
        check_inputs(1);               // Input-Abfrage
      }
      else if ((!strncasecmp(x1_buf, "VER?", 4)) && (x1_len >= 4))
      {
        x1_printf(VERSION);            // Version ausgeben
      }
      else if ((!strncasecmp(x1_buf, "UBM?", 4)) && (x1_len >= 4))
      {
        char buf[50];
        float volt = UMESS.read() * 36.3f;
        sprintf(buf, "UBM:%2.1fV\r", volt);     // Spannung ausgeben
        x1_printf(buf);
      }
      else if ((!strncasecmp(x1_buf, "X2:", 3)) && (x1_len >= 4))
      {
        x2.printf(&x1_buf[3]);         // Ausgabe auf X2 umleiten
      }
      else if ((!strncasecmp(x1_buf, "X3:", 3)) && (x1_len >= 4))
      {
        x3.printf(&x1_buf[3]);         // Ausgabe auf X3 umleiten
      }
      x1_new = 0;    
    }   
    check_inputs(0);                   // Eingaenge periodisch auf Veraenderung pruefen
    copy_eth_leds();                   // Ethernet-LED kopieren
  }
}
