// A class library for Azoteq IQS62x devices

// Copyright 2017 Azoteq. Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions: The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software. THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

// More info on IQS62x sensor ICs: http://www.azoteq.com/products/proxfusion?mbed

// ProxFusion 5-minute YouTube video: http://bit.ly/proxfusion-video

#include "IQS62x.h"
DigitalOut toggle(PB_3);

// constructor
IQS62xIO::IQS62xIO() :
    i2c( IQS_I2C_DATA_PIN, IQS_I2C_CLOCK_PIN ), // first run the constructor for mbed class I2C
    IQSready( IQS_READY_PIN )                   // first run the constructor for mbed class DigitalIn
{
    registers = I2CBuffer; // pointer to the receive buffer
    I2CErrorCount = 0; // reset I2C error count
    IQSframes = 0; // reset number of reads
    i2c.frequency( I2Cspeed ); // I2C clock frequency

    // this uses memory but is very handy in diagnostics
    memset(writeFlag,        0, I2CBufferSize); // a table to remember if we wrote to a register
    memset(lastWrite,        0, I2CBufferSize); // a table to remember what we wrote to a register
    memset(lastRead,         0, I2CBufferSize); // a table to remember what we read the previous read cycle
    memset(readChanges,      0, I2CBufferSize); // a table to mark up any registers that changed since the previous read
    memset(readChangesEver,  0, I2CBufferSize); // a table to mark up any registers that changed BUT never forget
    memset(writeChanges,     0, I2CBufferSize); // a table to mark up any registers that differ from what was written to it
}

// write a single byte to an IQS62x register
void IQS62xIO::writeRegister(int address, int data)
{
    toggle = !toggle;
    writeFlag[ address & 0xff ] = 1; // remember which registers we changed
    lastWrite[ address & 0xff ] = data & 0xff; // remember what we wrote
    char twoBytes [2];
    int numberOfBytes = 2;
    twoBytes[0] = address & 0xff;
    twoBytes[1] = data & 0xff;
    waitForIqsReady();
    if(0!=i2c.write(I2C_ADR,twoBytes,numberOfBytes,false))
        I2CErrorCount++;
}

// configure the IQS62x
#include "deviceType.h" // to check if we should override the configuration definition
#ifndef OVERRIDE_CONFIGURE
// if there is no override for configure() then we use this simple version
void IQS62xIO::configure()
{
    writeRegister(0xd0,0x40); // simplest config : just acknowledge/clear the reset flag
}
#endif

// read N registers, starting at provided offset
void IQS62xIO::readIqsRegisters(int start, int count)
{
    memset(I2CBuffer,0x55,I2CBufferSize); // "clear" i2c receive buffer
    IQSframes++; // count frames
    waitForIqsReady();
    char i2c_start_address [1];
    i2c_start_address[0] = start & 0xff;
    int numberOfBytes = 1;
    // write start address to the IQS62x address register
    if(0!=i2c.write(I2C_ADR,i2c_start_address,numberOfBytes,false))
        I2CErrorCount++;
    waitForIqsReady();
    numberOfBytes = count % I2CBufferSize;
    // read register values into a buffer
    if(0!=i2c.read(I2C_ADR,I2CBuffer,numberOfBytes,false))
        I2CErrorCount++;

#define DONT_CHECK_FOR_IQS_CHANGES 1
#ifndef DONT_CHECK_FOR_IQS_CHANGES
    // this is optional but handy diagnostics
    // we build six tables:
    //writeFlag -       a table to remember if we wrote to a register
    //lastWrite -       a table to remember what we wrote to a register
    //lastRead -        a table to remember what we read the previous read cycle
    //readChanges -     a table to mark up any registers that changed since the previous read
    //readChangesEver - a table to mark up any registers that changed BUT never forget
    //writeChanges -    a table to mark up any registers that differ from what was written to it
    if (IQSframes == 1) memcpy(lastRead, I2CBuffer,I2CBufferSize); // if this is the first frame init the history buffer
    if (IQSframes < 20) memset(readChangesEver,0,I2CBufferSize); // clear this until we have had 20 full reads
    for(int i=0; i<I2CBufferSize; i++) {
        if (writeFlag[i] == 1) { // only if we previously wrote to this register we check it
            if( I2CBuffer[i] != lastWrite[i] ) 
            writeChanges[i]=1; // if register different from what we wrote, mark it
        }
        if ( I2CBuffer[i] != lastRead[i] ) {
            readChanges[i]=20; // if register differs from previous read, highlight it for 20 read cycles
            if (readChangesEver[i] < 127) readChangesEver[i]++; // count changes up to 127 and never forget
        } else {
            if (readChanges[i] > 0) readChanges[i]--; // bring it closer to zero
        }
    }
    memcpy(lastRead, I2CBuffer, I2CBufferSize); // preserve history for next round
#endif
}

// wait for IQS62x to provide a ready signal (low) on IQS62x_ready pin
void IQS62xIO::waitForIqsReady()
{
    int timeout=0;
    while (1) {
        if(IQSready==1) break;
        if (timeout++ > 1000000) goto fatal_error;
    }
    timeout=0;
    while (1) {
        if(IQSready==0) break;
        if (timeout++ > 1000000) goto fatal_error;
    }
    return;; 
fatal_error:
    error ("Fatal Error: IQS62x ready pin is not toggling");
}

// return one of the tables of changes that we maintain
char * IQS62xIO::getTable( int option )
{
    switch ( option ) {
        case 1 : return  readChangesEver; // a table to flag any register that ever changed
        case 2 : return  readChanges;     // a table to flag any register that changed in the last 20 reads
        case 3 : return  writeFlag;       // a table to flag any register we wrote to or initialized
        case 4 : return  writeChanges;    // a table to flag any register we wrote to but then its value changed
        default: return NULL;; 
    }
}