// A class library to display Azoteq IQS622 registers on a terminal

// Copyright 2017 Azoteq. Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions: The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software. THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

// More info on IQS622 sensor IC: http://bit.ly/IQS622-info

// IQS622 1-minute youtube video: N.A.

#include "IQS622DisplayTerminal.h"

// constructor
#if defined(TARGET_TEENSY3_1) || defined (TARGET_TEENSY3_2) || IQS_USE_USBSERIAL
IQS622Display::IQS622Display() : USBSerial() // use our own USB device stack
#else
IQS622Display::IQS622Display() : Serial(USBTX,USBRX) // use mbed default serial port
#endif

{
    frameCounter=0;
    baud(DISPLAY_BAUD_RATE);
}

#if defined(TARGET_TEENSY3_1) || defined (TARGET_TEENSY3_2) || IQS_USE_USBSERIAL
void IQS622Display::baud(int baudRate) {}
#endif

// display a startup message to serial port
void IQS622Display::helloMessage(bool waitForUser)
{
    puts("\x1b[2J  \x1b[?25l  \x1b[H"); // ANSII/VT100 codes to clear screen, invisible cursor, home cursor
    printf("           IQS622 Register Display\r\n\r\n");
    printf("To get a smooth screen refresh effect, use a terminal program that supports ANSI/VT100 escape codes such as Tera Term.\r\n\r\n");
    printf("Handy hint - In many terminal programs, Alt-B (break) will reset your mbed Board.\r\n\r\n");
    printf("Press any key to continue...\r\n");
    if ( waitForUser ) while( ! readable() ); // wait for keypress to continue
    puts("\x1b[2J  \x1b[?25l  \x1b[H"); // ANSII/VT100 codes to clear screen, invisible cursor, home cursor
}

// show headings and I2C error count
void IQS622Display::showStatus(int I2Cspeed, int I2CErrorCount)
{
    frameCounter++;
    puts("\x1b[H"); // ANSI/VT100 command for cursor home
    printf("\t");
    printf("\t IQS622 Register Display\r\n\r\n"); // heading
    printf("\t Frame number %06d", frameCounter);
    printf("\t I2C Speed %dk", I2Cspeed/1000);
    printf("\t I2C Errors %d", I2CErrorCount);
}

// dump one line of text from the buffer with VT100 color formatting
void IQS622Display::showLine(char * buffer, char * color, int startbyte, int endbyte)
{
    for (int i=startbyte; i<=endbyte; i++) {
        if ( color == NULL || color[i] == 0 )
            printf("%02x ", buffer[i]);
        else
#define USE_IQS_COLOR_DISPLAY
#ifndef DONT_USE_IQS_COLOR_DISPLAY
            printf("\x1b[32m%02x\x1b[30m ", buffer[i]); // print out in green (ANSI VT100 code)
#else
            printf("%02x ", buffer[i]); // print out in black & white
#endif
    }
}

// formatted hex display of IQS622 registers
void IQS622Display::showRegisters(char * buffer,char * color, bool showAllRegisters)
{
    bool d = showAllRegisters; // if true show all 16 registers per line
    printf("\r\n\r\n Device ID & Version (42) [00] ");    showLine(buffer,color,0x00,d?0x0f:0x02);
    printf("\r\n\r\n         Events and Flags [10] ");    showLine(buffer,color,0x10,d?0x1f:0x1a);
    printf("\r\n\r\n   Channel 0-6 Raw Values [20] ");    showLine(buffer,color,0x20,d?0x2f:0x2d);
    printf("\r\n\r\n   Long Term Average Data [30] ");    showLine(buffer,color,0x30,d?0x3f:0x33);
    printf("\r\n\r\n      Proxfusion Settings [40] ");    showLine(buffer,color,0x40,d?0x4f:0x4d);
    printf("\r\n\r\n    Proxfusion Thresholds [50] ");    showLine(buffer,color,0x50,d?0x5f:0x54);
    printf("\r\n\r\n           SAR Thresholds [60] ");    showLine(buffer,color,0x60,d?0x6f:0x65);
    printf("\r\n\r\n Light/IR Sensor Settings [70] ");    showLine(buffer,color,0x70,d?0x7f:0x75);
    printf("\r\n\r\n    IR Threshold Settings [90] ");    showLine(buffer,color,0x90,d?0x9f:0x92);
    printf("\r\n\r\n   Hall Switch Thresholds [A0] ");    showLine(buffer,color,0xa0,d?0xaf:0xa3);
    printf("\r\n\r\n   Temperature Thresholds [B0] ");    showLine(buffer,color,0xb0,d?0xbf:0xb2);
    printf("\r\n\r\n  Device & Power Settings [D0] ");    showLine(buffer,color,0xd0,d?0xdf:0xd7);
    printf("\r\n\r\n");
}