/* =====================================================================
   Copyright © 2016, Avnet (R)

   Contributors:
     * James M Flynn, www.em.avnet.com 
 
   Licensed under the Apache License, Version 2.0 (the "License"); 
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, 
   software distributed under the License is distributed on an 
   "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
   either express or implied. See the License for the specific 
   language governing permissions and limitations under the License.

    @file          main.cpp / WNCInterface_HTTPS_Example
    @version       1.0
    @date          Dec 2016

======================================================================== */

#include "mbed.h"
 
#include "WNCInterface.h"

#include "Socket/Socket.h"
#include "Socket/TCPSocketConnection.h"
#include "Socket/UDPSocket.h"

#define DEBUG
#define MBED_PLATFORM

#include "HTTPClient.h"

#define STREAM_CNT  15          //when we test streaming, this is how many times to stream the string
#define STR_SIZE    125*(STREAM_CNT+1) //use a fixed size string buffer based on the streaming data count
#define CRLF    "\n\r"

//
// This example is setup to use MBED OS (5.2).  It sets up a thread to call the different tests
// because mbed tls is stack intensive and there is no way to modify the default stack size for
// main.  So when we create the test task, give it a large stack (x4 the default size).  After 
// this the main() task will just spin.
//

void https_test_thread(void);

int main() {
    Thread http_test(osPriorityNormal, DEFAULT_STACK_SIZE*4, NULL);

    printf("Testing HTTPClient and HTTPSClient using the WNCInterface & Socket software" CRLF);
    
    http_test.start(https_test_thread);
    while (true) {
        osDelay(500);
    }
}

//
// The two test functions do the same set of tests, the first one uses standard HTTP methods while
// the second test uses HTTP methods in conjunction with a SSL/TLS connection to verify certificates.
//

void test_http(void);       //function tests the standard HTTPClient class
void test_https(void);      //function to test the HTTPSClient class
        
void https_test_thread(void) {
    int ret;

    WNCInterface wnc;
    
    ret = wnc.init();  
    printf("WNC Module %s initialized (%02X)." CRLF, ret?"IS":"IS NOT", ret);
    if( !ret ) {
        printf(" - - - - - - - ALL DONE - - - - - - - " CRLF);
        while(1);
        }
        
    ret = wnc.connect();                 
    printf("IP Address: %s " CRLF CRLF, wnc.getIPAddress());

    test_http();
    test_https();
    
    wnc.disconnect();
    printf(" - - - - - - - ALL DONE - - - - - - - " CRLF);
    while(1) {}
}



//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
// test the HTTP client class
//
void test_http(void) {
    HTTPClient  http;
    char str[STR_SIZE];
    int ret;
    
    printf(">>>>>>>>>>>><<<<<<<<<<<<" CRLF);
    printf(">>>  TEST HTTPClient <<< "CRLF);
    printf(">>>>>>>>>>>><<<<<<<<<<<<" CRLF CRLF);

    //GET data
    printf(" ** Fetch a page... **" CRLF);

    ret = http.get("https://developer.mbed.org/media/uploads/mbed_official/hello.txt", str, STR_SIZE);
    if (!ret) {
      printf("Page fetched successfully - read %d characters" CRLF, strlen(str));
      printf("<----->" CRLF "Result: %s" CRLF "<----->" CRLF, str); 
      }
    else 
      printf("Error - ret = %d - HTTP return code = %d" CRLF, ret, http.getHTTPResponseCode());

    //POST data
    HTTPMap map;
    HTTPText inText(str, STR_SIZE);

    map.put("Hello", "World");
    map.put("test", "1234");

    printf(CRLF CRLF " ** Post data... **" CRLF);
    ret = http.post("http://httpbin.org/post", map, &inText);
    if (!ret) {
      printf("Executed POST successfully - read %d characters" CRLF, strlen(str));
      printf("<----->" CRLF );
      printf("Result: %s" CRLF "<----->" CRLF, str);
      }
    else 
      printf("Error - ret = %d - HTTP return code = %d" CRLF, ret, http.getHTTPResponseCode());
    
    //PUT data
    
    strcpy(str, "This is a PUT test!");
    HTTPText outText(str);    
    printf(CRLF CRLF " ** Put data... **" CRLF);

    ret = http.put("http://httpbin.org/put", outText, &inText);
    if (!ret) {
      printf("Executed PUT successfully - read %d characters" CRLF, strlen(str));
      printf("<----->" CRLF );
      printf("Result: %s" CRLF "<----->" CRLF, str); 
      }
    else 
      printf("Error - ret = %d - HTTP return code = %d" CRLF, ret, http.getHTTPResponseCode());
    
    //DELETE data
    printf(CRLF CRLF " ** Delete data... **" CRLF);
    ret = http.del("http://httpbin.org/delete", &inText);
    if (!ret) {
      printf("Executed DELETE successfully - read %d characters" CRLF, strlen(str));
      printf("<----->" CRLF );
      printf("Result: %s" CRLF "<----->" CRLF, str);
      }
    else 
      printf("Error - ret = %d - HTTP return code = %d" CRLF, ret, http.getHTTPResponseCode());

    printf(CRLF CRLF " ** HTTP:stream data " INTSTR(STREAM_CNT) " times... **" CRLF);
    if( (ret=http.get("http://httpbin.org:80/stream/" INTSTR(STREAM_CNT), str,sizeof(str))) == HTTP_OK) {
        printf(CRLF "STREAM successfull - returned %d characters" CRLF, strlen(str));
        printf("<----->" CRLF "Result:" CRLF "%s" CRLF "<----->" CRLF CRLF, str);
        }
    else
        printf(CRLF "STREAM FAILED!, returned %d" CRLF, ret);
        
        
    printf(CRLF CRLF ">>>>HTTP:Status..." CRLF);
    ret = http.get("http://httpbin.org/get?show_env=1", str, STR_SIZE);
    if (!ret) {
      printf("Executed STATUS successfully - read %d characters" CRLF, strlen(str));
      printf("<----->" CRLF );
      printf("Result: %s" CRLF "<----->" CRLF, str);
      }
    else {
      printf("Error - ret = %d - HTTP return code = -0x%04X" CRLF, ret, -http.getHTTPResponseCode());
      }

}


//- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
// test the HTTPS client class
//
void test_https(void) {
    HTTPSClient *https = new HTTPSClient;  //each time you call a Host, it sets the CA Certificate
                                           //for that host.  Because we call two different hosts it
                                           //is easier to create the class on the heap rather than
                                           //statically...
    char str[STR_SIZE];
    int ret;
    
    https->setHost("developer.mbed.org");
    https->addRootCACertificate(SSL_CA_PEM);  //this is the CA certificate for mbed.org

    printf(">>>>>>>>>>>><<<<<<<<<<<<" CRLF);
    printf(">>> TEST HTTPSClient <<<" CRLF);
    printf(">>>>>>>>>>>><<<<<<<<<<<<" CRLF CRLF);
 
    //GET data

    memset(str,0x00,STR_SIZE);
    printf(CRLF " ** HTTPS:Fetch a page... **" CRLF);
                                                   
    ret = https->get("https://developer.mbed.org/media/uploads/mbed_official/hello.txt", str, STR_SIZE);
    if (!ret){
      printf("Page fetched successfully - read %d characters" CRLF, strlen(str));
      printf("<----->" CRLF "Result: %s" CRLF "<----->" CRLF, str); 
      }
    else 
      printf("Error - ret = %d - HTTP return code = -0x%04X" CRLF, ret, -https->getHTTPResponseCode());

    delete https;                                 //go ahead and delete the mbed.org hppts client 
    https = new HTTPSClient;                      //and create one for httbin.org
    https->setHost("httpbin.org");
    https->addRootCACertificate(SSL_CA_HTTPBIN);  //set the CA certificate for httpbin.org

    HTTPMap map;
    HTTPText inText(str, STR_SIZE);

    //POST data
    map.put("https Testing", "https doing a Post");
    printf(CRLF CRLF ">>>>HTTPS:Post data..." CRLF);
    ret = https->post("https://httpbin.org:443/post", map, &inText);
    if (!ret) {
      printf("Executed POST successfully - read %d characters" CRLF, strlen(str));
      printf("<----->" CRLF );
      printf("Result: %s" CRLF "<----->" CRLF, str);
      }
    else 
      printf("Error - ret = %d - HTTP return code = -0x%04X" CRLF, ret, -https->getHTTPResponseCode());

    //PUT data
    memset(str,0x00,STR_SIZE);
    strcpy(str, "This is an HTTPS PUT test!");
    HTTPText outText(str);    
    printf(CRLF CRLF ">>>>HTTPS:Put data..." CRLF);
    ret = https->put("https://httpbin.org:443/put", outText, &inText);
    if (!ret) {
      printf("Executed PUT successfully - read %d characters" CRLF, strlen(str));
      printf("<----->" CRLF );
      printf("Result: %s" CRLF "<----->" CRLF, str); 
      }
    else 
      printf("Error - ret = %d - HTTP return code = -0x%04X" CRLF, ret, -https->getHTTPResponseCode());
   
    printf(CRLF CRLF ">>>>HTTPS:stream data..." CRLF);
    if( (ret=https->get("https://httpbin.org:443/stream/" INTSTR(STREAM_CNT), str, STR_SIZE)) == HTTP_OK) {
        printf(CRLF "STREAM successfull - returned %d characters" CRLF, strlen(str));
        printf("<----->" CRLF "Result:" CRLF "%s" CRLF "<----->" CRLF, str);
        }
    else
        printf(CRLF "STREAM FAILED!, returned %d - HTTP return code = -x%04X" CRLF, ret, -https->getHTTPResponseCode());

    //DELETE data
    printf(CRLF CRLF ">>>>HTTPS:Delete data..." CRLF);
    ret = https->del("https://httpbin.org/delete", &inText);
    if (!ret) {
      printf("Executed DELETE successfully - read %d characters" CRLF, strlen(str));
      printf("<----->" CRLF );
      printf("Result: %s" CRLF "<----->" CRLF, str);
      }
    else {
      printf("Error - ret = %d - HTTP return code = -0x%04X" CRLF, ret, -https->getHTTPResponseCode());
      }
   
    printf(CRLF CRLF ">>>>HTTPS:Status..." CRLF);
    ret = https->get("https://httpbin.org:443/get?show_env=1", str, STR_SIZE);
    if (!ret) {
      printf("Executed STATUS successfully - read %d characters" CRLF, strlen(str));
      printf("<----->" CRLF );
      printf("Result: %s" CRLF "<----->" CRLF, str);
      }
    else {
      printf("Error - ret = %d - HTTP return code = -0x%04X" CRLF, ret, -https->getHTTPResponseCode());
      }

    delete https;  //all done, delete the httpbin.org https client
}

