/***
*   xbee.h - en-tete de lib xbee - fouj1807 - mcgj2701
*/

#pragma once

#include "mbed.h"
#include "BufferedSerial.h"

// xbee serial port object
extern BufferedSerial xbee;

// Type de frames possible dans le mode API
typedef enum {
    ATCommand = 0x08,           // Commande AT
    ATCommandQueue = 0x09,
    TransmitRequest = 0x10,     // Requete d'envoi de donnees
    ExplicitAddressing = 0x11,
    RemoteCommandRequest = 0x17,// Commande AT a distance
    CreateSourceRoute = 0x21,
    ATCommandResponse = 0x88,   
    ModemStatus = 0x8A,
    ZigBeeTransmitStatus = 0x8B,
    ZigBeeReceivePacket = 0x90,
    ZigBeeExplicitRxIndicator = 0x91,
    ZigBeeIODataSampleRxIndicator = 0x92,
    XBeeSensorReadIndicator = 0x94,
    NodeIndentificationIndicator = 0x95,
    RemoteCommandResponse = 0x97,
    OverTheAirFirmwareUpdateStatus = 0xA0,
    RouteRecordIndicator = 0xA1,
    ManyToOneRouteRequestIndicator = 0xA3
} frame_type_t;

// Structure d'addressage du zigbee (64-bit)
typedef struct {
    char addr_0;    // MSB
    char addr_1;
    char addr_2;
    char addr_3;

    char addr_4;
    char addr_5;
    char addr_6;
    char addr_7;    // LSB
} zigbee_addr_64_t;

// Structure d'addressage du zigbee (16-bit)
typedef struct {
    char addr_msb;
    char addr_lsb;
} zigbee_addr_16_t;

#define MAX_FRAME_SIZE 64

typedef struct {
    int length;
    char buffer[MAX_FRAME_SIZE];
} frame_t;

typedef enum {
    wait_delimiter,
    read_length_msb,
    read_length_lsb,
    read_frame_specific,
    read_checksum
} frame_recv_state_t;

// Position du frame delimiter
#define FRAME_DELIMITER_POS 0
// Length position MSB
#define FRAME_LEN_MSB_POS 1
// Length position LSB
#define FRAME_LEN_LSB_POS 2

// position Checksum 
#define FRAME_CHEKSUM_POS 3 + frame_data_size

// position Frame type 
#define API_FRAME_TYPE_POS 3
// position Frame ID
#define API_FRAME_ID_POS 4

// position Transmit req 64-bit address 
#define TRANSMIT_REQ_ADDR64_MSB_POS 5
// position Transmit req 16-bit address
#define TRANSMIT_REQ_ADDR16_MSB_POS 13
// position Transmit req radius
#define TRANSMIT_REQ_RADIUS_POS 15

// position de depart du data dans un transmit request
#define TRANSMIT_REQ_DATA_START 17
// taille du header de transmit request
#define TRANSMIT_REQ_OVERHEAD_LENGTH 14

#define REMOTE_AT_OVERHEAD_LENGTH 13
#define AT_COMMAND_OVERHEAD_LENGTH 4

#define AT_COMMAND_DIO4_MSB 'D'
#define AT_COMMAND_DIO4_LSB '4'
#define AT_COMMAND_DIO_OUT_LOW 0x4
#define AT_COMMAND_DIO_OUT_HIGH 0x5

/*
* Macro permettant de declarer une address 64-bit ou 16-bit reservee
* Usage : 
*               DECLARE_ADDR16_UNKNOWN_OR_BCAST // declaration de l'addresse
*               USE_ADDR16_UNKNOWN_OR_BCAST // usage de l'addresse
*/
#define DECLARE_ADDR16_UNKNOWN_OR_BCAST zigbee_addr_16_t def_addr16_bcast = { 0xFF, 0xFE };
#define USE_ADDR16_UNKNOWN_OR_BCAST def_addr16_bcast
#define DECLARE_ADDR64_COORD zigbee_addr_64_t def_addr64_coord = { 0, 0, 0, 0, 0, 0, 0, 0 };
#define USE_ADDR64_COORD def_addr64_coord
#define DECLARE_ADDR64_BCAST zigbee_addr_64_t def_addr64_bcast = { 0, 0, 0, 0, 0, 0, 0xFF, 0xFE };
#define USE_ADDR64_BCAST def_addr64_bcast

// Zigbee delimiter value
#define ZIGBEE_DELIMITER 0x7E

// Transmit/receive buffer space
extern char TransmitBuffer[2048];
extern char ReceiveBuffer[2048];

// Compare if two 64bit addresses are equal
bool addr_64_equal(zigbee_addr_64_t addr1, zigbee_addr_64_t addr2);

// Init xbee communication
void xbee_init();

// insere le api frame header dans le buffer
void build_api_frame_header(int frame_data_size);
// insere le api frame cheksum dans le buffer
void build_api_frame_checksum(int frame_data_size);

// Set le type d'api frame
void set_api_frame_type(frame_type_t frame_type);
// set le ID du frame
void set_api_frame_id(int id);

// Set at command not implemented
void set_at_command(int at_command);
// Set l'address 64 bit 
void set_64_addr(zigbee_addr_64_t address);
// Set l'address 16 bit 
void set_16_addr(zigbee_addr_16_t address);

// Set le radius
void set_broadcast_radius(int position, char radius);
// Set les options a la position indiquée
void set_transmit_request_options(int position, char options);
// Set le data du transmit request
void set_transmit_data(int start_pos, char* data, int data_length);
// Fonctions de transmission variées pour les différentes situations
bool transmit_request_64(char* buffer, int data_length, char options, zigbee_addr_64_t destination);
bool transmit_request_16(char* buffer, int data_length, char options, zigbee_addr_16_t destination);
bool transmit_request(char* buffer, int data_length, char options, zigbee_addr_64_t destination64, zigbee_addr_16_t destination16);

void at_command_query(char msb, char lsb);
void at_command_set(char msb, char lsb, char* parameter, int parameter_length);

// Envoie d'un remote at command (set ou query) en broadcast
void remote_at_command_query(char msb, char lsb, char options, zigbee_addr_64_t destination);
void remote_at_command_set(char msb, char lsb, char parameter, char options, zigbee_addr_64_t destination);

void transmit(int packet_length);
bool receive(frame_t* buffer);

