#include <cstddef>
#include <string.h>
#include "defines.hpp"
#include "configs.hpp"
#include "anthem.hpp"

#include <stdio.h>

LEDModule::LEDModule(void)
 : _rowNum(MODULE_HEIGHT),
   _colNum(MODULE_WIDTH)
{
    int i = 0;

    // Read in 3 things from the EEPROM:
    //      - LED driver to pixel mapping
    //      - Scan group to pixel mapping
    //      - LED driver channel to pixel mapping

    // Also read in other auxillary data from EEPROM:
    //      - Module brightness

    _moduleBrightness  = 1.0f; // FIXME get from EEPROM

    // FIXME where does this come from?  It will change depending on day/night correct?
    _displayBrightness = 1.0f;

    //--------------------------------------------
    //               SCAN GROUPS
    //--------------------------------------------
    ScanGroup *lastScanGroup = NULL;
    for (i = 0; i < NUM_SCANGROUPS; i++) {
        if (i == 0) {
            _currentScanGroup = &_scanGroupArray[i];
        }

        // Tie the last to this one
        if (lastScanGroup != NULL) {
            lastScanGroup->_nextScanGroup = &_scanGroupArray[i];
        }

        // If we are the last, tie it to the first
        if (i == (NUM_SCANGROUPS-1)) {
            _scanGroupArray[i]._nextScanGroup = &_scanGroupArray[0];
        }

        lastScanGroup = &_scanGroupArray[i];
    }
    //--------------------------------------------
    //--------------------------------------------


    //--------------------------------------------
    //               LED DRIVERS
    //--------------------------------------------
    ToshibaTC62D723 *lastDriver = NULL;
    for (i = 0; i < NUM_LED_DRIVERS; i++) {
        if (i == 0) {
            _dataOutDriver = &_ledDriverArray[i];
        }

        // Tie the last to this one
        if (lastDriver != NULL) {
            lastDriver->_dataOut = &_ledDriverArray[i];
            _ledDriverArray[i]._dataIn = lastDriver;
        }

        if (i == (NUM_LED_DRIVERS-1)) {
            _dataInDriver = &_ledDriverArray[i];
        }

        lastDriver = &_ledDriverArray[i];
    }
    //--------------------------------------------
    //--------------------------------------------


    //--------------------------------------------
    //               LED PIXELS
    //--------------------------------------------
    int x = 0;
    int y = 0;
    for (y = 0; y < MODULE_WIDTH; y++) {
        for (x = 0; x < MODULE_HEIGHT; x++) {
            // FIXME Get the calibration factor from EEPROM
            _pixel[x][y].setCalibrationFactor(1.0f);
            // FIXME Get the calibration factor from EEPROM
            _pixel[x][y].  _redLED.setCalibrationFactor(1.0f);
            _pixel[x][y]._greenLED.setCalibrationFactor(1.0f);
            _pixel[x][y]. _blueLED.setCalibrationFactor(1.0f);
            _pixel[x][y].  _redLED._ledDriver = &_ledDriverArray[ g_driverChain[g_pix2Drv[x][y].  redDrv] ];
            _pixel[x][y]._greenLED._ledDriver = &_ledDriverArray[ g_driverChain[g_pix2Drv[x][y].greenDrv] ];
            _pixel[x][y]. _blueLED._ledDriver = &_ledDriverArray[ g_driverChain[g_pix2Drv[x][y]. blueDrv] ];
            _pixel[x][y].  _redLED.setDriverChannelNum(g_pix2DrvChan[x][y]);
            _pixel[x][y]._greenLED.setDriverChannelNum(g_pix2DrvChan[x][y]);
            _pixel[x][y]. _blueLED.setDriverChannelNum(g_pix2DrvChan[x][y]);

            // Nothing needs to change for column-based vs row-based scangroups
            _scanGroupArray[ g_pix2ScanGroup[x][y] ].addLED( &(_pixel[x][y].  _redLED) );
            _scanGroupArray[ g_pix2ScanGroup[x][y] ].addLED( &(_pixel[x][y]._greenLED) );
            _scanGroupArray[ g_pix2ScanGroup[x][y] ].addLED( &(_pixel[x][y]. _blueLED) );
        }
    }
    //--------------------------------------------
    //--------------------------------------------
}


LEDModule::~LEDModule(void)
{
}

void LEDModule::displayImage(void)
{
    int i = 0;

    adjustImage();

    for (i = 0; i < NUM_SCANGROUPS; i++) {
        BAGINFO3("\nScanGroup[%d]", i);
        _scanGroupArray[i].mapLEDsToDrivers();
        shiftBrightnessDataIn();
        enableNextScanGroup();
        ToshibaTC62D723::outputBrightnessDataToLEDs();
    }
}


void LEDModule::adjustImage(void)
{
    int x = 0;
    int y = 0;
    PixelColor oldColor;
    for (y = 0; y < MODULE_WIDTH; y++) {
        for (x = 0; x < MODULE_HEIGHT; x++) {
            oldColor = _sourceImage.getPixel(x, y);
            _pixel[x][y]._redLED.setBrightness
                                (
                                    oldColor._r * 256
                                    * _displayBrightness
                                    * _moduleBrightness
                                    * _pixel[x][y].getCalibrationFactor()
                                    * _pixel[x][y]._redLED.getCalibrationFactor()
                                );
            _pixel[x][y]._greenLED.setBrightness
                                (
                                    oldColor._g * 256
                                    * _displayBrightness
                                    * _moduleBrightness
                                    * _pixel[x][y].getCalibrationFactor()
                                    * _pixel[x][y]._greenLED.getCalibrationFactor()
                                );
            _pixel[x][y]._blueLED.setBrightness
                                (
                                    oldColor._b * 256
                                    * _displayBrightness
                                    * _moduleBrightness
                                    * _pixel[x][y].getCalibrationFactor()
                                    * _pixel[x][y]._blueLED.getCalibrationFactor()
                                );
        }
    }
}


void LEDModule::shiftBrightnessDataIn(void)
{
    int i = 0; // Driver index
    int chan = 0; // Channel
    for (i = NUM_LED_DRIVERS-1; i >= 0; i--) {
        BAGINFO3("\nDRV[%02d]", g_dbgDrvChain[i]);
        for (chan = (ToshibaTC62D723::NUM_CHANNELS)-1; chan >= 0; chan--) {
            ToshibaTC62D723::shiftBrightnessDataIn(_ledDriverArray[i]._channelBrightness[chan]);
        }
    }
}


void LEDModule::enableNextScanGroup(void)
{
    ScanGroup *lastScanGroup = NULL;
    if (_currentScanGroup != NULL) {
        lastScanGroup = _currentScanGroup;
        if (_currentScanGroup->_nextScanGroup != NULL) {
            _currentScanGroup = _currentScanGroup->_nextScanGroup;
            setPinToValue(lastScanGroup->getTransistorMCUPinNum(), LOW);
            setPinToValue(_currentScanGroup->getTransistorMCUPinNum(), HIGH);
        }
    }
}


void LEDModule::setPinToValue(int p_pin, int p_val)
{
    // TODO
    // Set a GPIO value here
}


ScanGroup::ScanGroup(void)
 : _nextScanGroup(NULL),
   _transistorMCUPinNum(0),
   _index(0)
{
    memset(_LEDs, 0, sizeof(LED*) * LEDS_PER_SCANGROUP);
}


ScanGroup::~ScanGroup(void)
{
    // Its good practice to clear memory in the destructor
    memset(_LEDs, 0, sizeof(LED*) * LEDS_PER_SCANGROUP);
}


// For all individual LEDs in this scangroup, set its driver's
// brightness for the appropriate channel.
void ScanGroup::mapLEDsToDrivers(void)
{
    int i = 0;
    for (i = 0; i < LEDS_PER_SCANGROUP; i++) {
        _LEDs[i]->_ledDriver->_channelBrightness[_LEDs[i]->getDriverChannelNum()] = _LEDs[i]->getBrightness();
    }
}


void ScanGroup::addLED(LED *p_led)
{
    if ( (_index < LEDS_PER_SCANGROUP) &&
         (p_led != NULL)
       )
    {
        _LEDs[_index] = p_led;
        _index++;
    }
}

int ScanGroup::getTransistorMCUPinNum(void)
{
    // TODO
    return 0;
}


LEDPixel::LEDPixel(void)
 : _calibrationFactor(0.0f)
{
}

LEDPixel::~LEDPixel(void)
{
}

float LEDPixel::getCalibrationFactor(void)
{
    return _calibrationFactor;
}

void LEDPixel::setCalibrationFactor(float p_cal)
{
    _calibrationFactor = p_cal;
}


LED::LED(void)
 : _ledDriver(NULL),
   _calibrationFactor(0.0f),
   _driverChannelNum(0),
   _brightness(0)
{
}

LED::~LED(void)
{
}

float LED::getCalibrationFactor(void)
{
    return _calibrationFactor;
}

void LED::setCalibrationFactor(float p_val)
{
    _calibrationFactor = p_val;
}

uint8_t LED::getDriverChannelNum(void)
{
    return _driverChannelNum;
}

void LED::setDriverChannelNum(uint8_t p_val)
{
    _driverChannelNum = p_val;
}

uint16_t LED::getBrightness(void)
{
    return _brightness;
}

void LED::setBrightness(uint16_t p_val)
{
    _brightness = p_val;
}


Image::Image(void)
{
}
Image::~Image(void)
{
}
PixelColor Image::getPixel(uint16_t x, uint16_t y)
{
    return _imgPixel[x][y];
}
void Image::setPixel(uint16_t x, uint16_t y, PixelColor p_pix)
{
    _imgPixel[x][y] = p_pix;
}


PixelColor::PixelColor(void)
 : _r(0), _g(0), _b(0)
{
}
PixelColor::~PixelColor(void)
{
}
