/***************************************************************************//**
 *   @file   adt7xxx_support.c
 *   @brief  Support Source File for adt7xxx
********************************************************************************
 * Copyright (c) 2021 Analog Devices, Inc.
 * All rights reserved.
 *
 * This software is proprietary to Analog Devices, Inc. and its licensors.
 * By using this software you agree to the terms of the associated
 * Analog Devices Software License Agreement.
*******************************************************************************/

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include <stdint.h>
#include "adt7420.h"
#include "error.h"
#include "adt7xxx_support.h"

/******************************************************************************/
/********************** Macros and Constants Definitions **********************/
/******************************************************************************/

/******************************************************************************/
/************************ Functions Definitions *******************************/
/******************************************************************************/

/***************************************************************************//**
 * @brief Reads the value of a register SPI/I2C.
 *
 * @param dev              - The device structure.
 * @param register_address - Address of the register.
 *
 * @return register_value  - Value of the register.
*******************************************************************************/
uint16_t adt7420_get_register_address_and_value(struct adt7420_dev *dev,
			    uint8_t register_address)
{
	uint16_t register_value = 0;

	//remap register map
	if(adt7420_is_spi(dev)) {
		switch (register_address) {
		case REG_TEMP:
			register_address = ADT7320_REG_TEMP;
			break;	 // Temperature value
		case REG_STATUS :
			register_address = ADT7320_REG_STATUS;
			break; // status info
		case REG_CONFIG :
			register_address = ADT7320_REG_CONFIG;
			break; // Configuration
		case REG_T_CRIT :
			register_address = ADT7320_REG_T_CRIT;
			break; // Temperature CRIT setpoint (147'C)
		case REG_HIST :
			register_address = ADT7320_REG_HIST;
			break;   // Temperature HYST setpoint (5'C)
		case REG_T_HIGH :
			register_address = ADT7320_REG_T_HIGH;
			break; // Temperature HIGH setpoint (64'C)
		case REG_T_LOW :
			register_address = ADT7320_REG_T_LOW;
			break;  // Temperature LOW setpoint (10'C)
		case REG_ID :
			register_address = ADT7320_REG_ID;
			break;	 // ID value
		}
		register_value = adt7420_get_register_value(dev, register_address);
	} else {
		switch (register_address) {
		case REG_TEMP: {
			register_value  = adt7420_get_register_value(dev,
					  ADT7420_REG_TEMP_MSB) << 8;
			register_value |= adt7420_get_register_value(dev, ADT7420_REG_TEMP_LSB);
			break;
		}
		case REG_STATUS:
			register_value = adt7420_get_register_value(dev, ADT7420_REG_STATUS);
			break;
		case REG_CONFIG:
			register_value = adt7420_get_register_value(dev, ADT7420_REG_CONFIG);
			break;
		case REG_T_HIGH: {
			register_value  = adt7420_get_register_value(dev,
					  ADT7420_REG_T_HIGH_MSB) << 8;
			register_value |= adt7420_get_register_value(dev, ADT7420_REG_T_HIGH_LSB);
			break;
		}
		case REG_T_LOW: {
			register_value  = adt7420_get_register_value(dev,
					  ADT7420_REG_T_LOW_MSB) << 8;
			register_value |= adt7420_get_register_value(dev, ADT7420_REG_T_LOW_LSB);
			break;
		}
		case REG_T_CRIT: {
			register_value  = adt7420_get_register_value(dev,
					  ADT7420_REG_T_CRIT_MSB) << 8;
			register_value |= adt7420_get_register_value(dev, ADT7420_REG_T_CRIT_LSB);
			break;
		}
		case REG_HIST:
			register_value = adt7420_get_register_value(dev, ADT7420_REG_HIST);
			break;
		case REG_ID:
			register_value = adt7420_get_register_value(dev, ADT7420_REG_ID);
			break;
		}
	}
	return register_value;
}


/**************************************************************************//**
 * @brief Configure write typev register based on communication interface.
 *
 * @param dev					- The device structure.
 * @param register_address		- Register type.
 *
 * @return  register_address	- Register Address.
******************************************************************************/
uint16_t configure_write_type_registers(struct adt7420_dev *dev,
					uint8_t register_address)
{
	if (adt7420_is_spi(dev)) {
		//simple address re-map for SPI devices
		switch(register_address) {
		case REG_TEMP:
			register_address = ADT7320_REG_TEMP;
			break; // Temperature value
		case REG_STATUS :
			register_address = ADT7320_REG_STATUS;
			break; // status info
		case REG_CONFIG :
			register_address = ADT7320_REG_CONFIG;
			break; // Configuration
		case REG_T_CRIT :
			register_address = ADT7320_REG_T_CRIT;
			break; // Temperature CRIT setpoint (147'C)
		case REG_HIST :
			register_address = ADT7320_REG_HIST;
			break; // Temperature HYST setpoint (5'C)
		case REG_T_HIGH :
			register_address = ADT7320_REG_T_HIGH;
			break; // Temperature HIGH setpoint (64'C)
		case REG_T_LOW :
			register_address = ADT7320_REG_T_LOW;
			break; // Temperature LOW setpoint (10'C)
		case REG_ID :
			register_address = ADT7320_REG_ID;
			break; // ID value
		}
	} else {
		//simple address re-map for I2Cdevices
		switch(register_address) {
		case REG_TEMP:
			register_address = ADT7420_REG_T_HIGH_MSB;
			break; // Temperature value
		case REG_STATUS :
			register_address = ADT7420_REG_STATUS;
			break; // status info
		case REG_CONFIG :
			register_address = ADT7420_REG_CONFIG;
			break; // Configuration
		case REG_T_CRIT :
			register_address = ADT7420_REG_T_CRIT_MSB;
			break; // Temperature CRIT setpoint (147'C)
		case REG_HIST :
			register_address = ADT7420_REG_HIST;
			break; // Temperature HYST setpoint (5'C)
		case REG_T_HIGH :
			register_address = ADT7420_REG_T_HIGH_MSB;
			break; // Temperature HIGH setpoint (64'C)
		case REG_T_LOW :
			register_address = ADT7420_REG_T_LOW_MSB;
			break; // Temperature LOW setpoint (10'C)
		case REG_ID :
			register_address = ADT7420_REG_ID;
			break; // ID value
		}
	}

	return register_address;
}

/**************************************************************************//**
 * @brief Write to a setpoint register.
 *
 * @param dev            - The device structure.
 * @param register_value - Command control bits.
 * @param data           - Data to be written in input register.
 *
 * @return  read_back_data - value read from register.
******************************************************************************/
uint8_t adt7420_wr_setpoint_reg(struct adt7420_dev *dev,
				uint8_t register_value,
				uint16_t data)
{
	uint16_t read_back_data = 0;
	uint8_t address, num_bytes;
	uint8_t data_buffer[3] = { 0, 0, 0 };

	switch (register_value) {
	case REG_T_CRIT:
	case REG_T_HIGH:
	case REG_T_LOW: {
		num_bytes = 3;
		data_buffer[0] = ((data & ADT7420_MSB_MASK) >> ADT7420_MSB_OFFSET);
		data_buffer[1] = data & ADT7420_LSB_MASK;
		break;
	}
	case REG_HIST: {
		num_bytes = 2;
		data_buffer[0] = data & ADT7420_LSB_MASK;
		break;
	}
	}

	address = configure_write_type_registers(dev, register_value);
	set_register_value(dev, address, num_bytes, data_buffer, ADT7420_MASK_SET_PT_REGISTER, 0x00);
	read_back_data  = adt7420_get_register_value(dev, address);

	if (register_value == REG_HIST) {
		data &= 0x000F; //msbits are all low for HIST register
		read_back_data &= 0x000F; //msbits are all low for HIST register
	}

	return read_back_data == data ? SUCCESS : FAILURE;
}

/***************************************************************************//**
 * @brief Sets the Fault Queue option for ADT7420/ADT7320.
 *
 * @param dev        - The device structure.
 * @param mode       - Fault Queue selection.
 *                     Example: 1 - 1 fault (default).
 *                            	2 - 2 faults.
 *			                    3 - 3 faults.
 *			                    4 - 4 faults.
 *
 * @return None.
*******************************************************************************/
void adt7420_set_fault_queue(struct adt7420_dev *dev,
			     uint8_t mode)
{
	uint8_t register_value[1] = { 0 };
	uint8_t address;

	if (adt7420_is_spi(dev)) {
		address = ADT7320_REG_CONFIG;
		register_value[0]  = adt7420_get_register_value(dev, ADT7320_REG_CONFIG);
	} else {
		address = ADT7420_REG_CONFIG;
		register_value[0]  = adt7420_get_register_value(dev, ADT7420_REG_CONFIG);
	}

	set_register_value(dev, address, 2, register_value,
			   ADT7420_CONFIG_FAULT_QUEUE(ADT7420_FAULT_QUEUE_4_FAULTS),
			   ADT7420_CONFIG_FAULT_QUEUE(mode));
}

/***************************************************************************//**
 * @brief Sets comparator/interrupt (CT/INT) mode for ADT7420/ADT7320.
 *
 * @param dev        - The device structure.
 * @param setting    - Mode selection.
 *                     Example: 0 - Interrupt (default).
 *                            	1 - Comparator.
 *
 *
 * @return None.
*******************************************************************************/
void adt7420_set_ct_int_mode(struct adt7420_dev *dev,
			     uint8_t setting)
{
	uint8_t register_value[1] = { 0 };
	uint8_t address;

	if (adt7420_is_spi(dev)) {
		address = ADT7320_REG_CONFIG;
		register_value[0]  = adt7420_get_register_value(dev, ADT7320_REG_CONFIG);
	} else {
		address = ADT7420_REG_CONFIG;
		register_value[0]  = adt7420_get_register_value(dev, ADT7420_REG_CONFIG);
	}

	set_register_value(dev, address,2 ,register_value, ADT7420_CONFIG_INT_CT_MODE,
			   (setting * ADT7420_CONFIG_INT_CT_MODE));
}

/***************************************************************************//**
 * @brief Sets output polarity for the pins CT/INT (Critical Temp - Over/Under Temp).
 *
 * @param dev        - The device structure.
 * @param polarity   - Polarity selection.
 *                     Example: 0 - Active Low (default).
 *                            	1 - Active High.
 *
 *
 * @return None.
*******************************************************************************/
void adt7420_set_ct_int_polarity(struct adt7420_dev *dev,
				 uint8_t polarity)
{
	uint8_t register_value[1] = { 0 };
	uint8_t address;
	uint8_t bit_mask, bit_value;

	if (adt7420_is_spi(dev)) {
		address = ADT7320_REG_CONFIG;
		register_value[0]  = adt7420_get_register_value(dev, ADT7320_REG_CONFIG);
	} else {
		address = ADT7420_REG_CONFIG;
		register_value[0]  = adt7420_get_register_value(dev, ADT7420_REG_CONFIG);
		register_value[0]  = adt7420_get_register_value(dev, ADT7420_REG_CONFIG);
	}

	bit_mask = (ADT7420_CONFIG_CT_POL) | (ADT7420_CONFIG_INT_POL);
	bit_value = (polarity * ADT7420_CONFIG_CT_POL) | (polarity *
			ADT7420_CONFIG_INT_POL);

	set_register_value(dev, address, 2, register_value, bit_mask, bit_value);
}
