/* Copyright (c) 2021 Analog Devices, Inc.  All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright notice,
  this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.
  - Modified versions of the software must be conspicuously marked as such.
  - This software is licensed solely and exclusively for use with processors/products
  manufactured by or for Analog Devices, Inc.
  - This software may not be combined or merged with other code in any manner
  that would cause the software to become subject to terms and conditions which
  differ from those listed here.
  - Neither the name of Analog Devices, Inc. nor the names of its contributors
  may be used to endorse or promote products derived from this software without
  specific prior written permission.
  - The use of this software may or may not infringe the patent rights of one or
  more patent holders.  This license does not release you from the requirement
  that you obtain separate licenses from these patent holders to use this software.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES, INC. AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
TITLE, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
NO EVENT SHALL ANALOG DEVICES, INC. OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, PUNITIVE OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, DAMAGES ARISING OUT OF CLAIMS OF INTELLECTUAL
PROPERTY RIGHTS INFRINGEMENT; PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

2021-01-10-7CBSD SLA
*/

/**
 * @file     calibrate.cpp
 * @brief    This file contains calibrate related APIs for sdpk1 host
 */

/*=============  I N C L U D E S   =============*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ADMX2001.h"
#include "ADMX2001_commands.h"
#include "calibrate.h"
#include "measure.h"

/*=============  D A T A T Y P E S  =============*/

/**  maximum length for buffer for storing measurement values */
#define MAX_CALIBRATE_BUFFER 512

/**  maximum length for buffer for string operation */
#define MAX_CAL_DISPLAY_LENGTH 128
/*=============  D A T A  =============*/
/**  buffer for string operation */
extern char displayString[MAX_CAL_DISPLAY_LENGTH];

extern Admx200xDev admx200xDev;
/** Array to store measurement values*/
double calibrateBuffer[MAX_CALIBRATE_BUFFER];
/** Array to store sweep points*/
extern double sweepPoints[MAX_CALIBRATE_BUFFER];

static void DisplayCalibratedValue(double displayVal1, double displayVal2,
                            double sweepVar);
                            
static int32_t ReadAndDisplayCalibratedValue();



/**
 * @brief  Command to perform various calibration types.
 * @param [in] calType - Type of Calibrate
 * @param [in] stdR - stdR value
 * @param [in] stdX - stdX value
 * @return Returns 0 for success or negative error code.
 */
int32_t Calibrate(CAL_TYPE calType, float stdR, float stdX)
{
    int32_t status = ADMX_STATUS_FAILED;
    uint8_t cmdID;
    uint16_t addr;
    uint32_t statReg;
    uint32_t data;
    uint32_t *pStdR = (uint32_t *)&stdR;
    uint32_t data1 = *pStdR;
    uint32_t *pStdX = (uint32_t *)&stdX;
    uint32_t data2 = *pStdX;
    uint32_t calStatus;
    /** Pointer to the handle*/
    Admx200xDev *pAdmx200x = &admx200xDev;

    cmdID = CMD_CALIBRATE;
    if (calType == CAL_TYPE_LOAD)
    {
        /** Send the RT data */
        addr = 4;
        status = Admx200xSendCmd(pAdmx200x, &cmdID, &addr, &data1, &statReg);
        if (status == ADMX_STATUS_SUCCESS)
        {
            status = (int32_t)(statReg & ADMX200X_STATUS_CODE_BITM);
            /** Send the XT data */
            addr = 5;
            status =
                Admx200xSendCmd(pAdmx200x, &cmdID, &addr, &data2, &statReg);
            if (status == ADMX_STATUS_SUCCESS)
            {
                status = (int32_t)(statReg & ADMX200X_STATUS_CODE_BITM);
                /** Send the Load Calibration command*/
                // This calibration will use the previously set Rt and Xt data
                addr = 3;
                status =
                    Admx200xSendCmd(pAdmx200x, &cmdID, &addr, &data, &statReg);
                if (status == ADMX_STATUS_SUCCESS)
                {
                    status = Admx200xWaitForStatus(
                        pAdmx200x, SPI_TIMEOUT,
                        ADMX200X_STATUS_MEASURE_DONE_BITM, &calStatus);
                }
                ReadAndDisplayCalibratedValue();
            }
        }
    }
    else if (calType == CAL_TYPE_NO_CAL)
    {
        status = ADMX_STATUS_FAILED;
    }
    else
    {
        if (calType == CAL_TYPE_SHORT)
        {
            addr = 1;
        }
        else
        {
            addr = 2;
        }
        status = Admx200xSendCmd(pAdmx200x, &cmdID, &addr, &data, &statReg);
        if (status == ADMX_STATUS_SUCCESS)
        {
            status = Admx200xWaitForStatus(pAdmx200x, SPI_TIMEOUT,
                                           ADMX200X_STATUS_MEASURE_DONE_BITM,
                                           &calStatus);
        }
        ReadAndDisplayCalibratedValue();
    }

    return status;
}

/**
 * @brief  Function to read and display calibrated values
 * @return Returns 0 for success or negative error code.
 */
int32_t ReadAndDisplayCalibratedValue()
{
    int32_t i;
    int32_t status = ADMX_STATUS_SUCCESS;

    Admx200xDev *pAdmx200x = &admx200xDev;
    uint8_t cmdId = 0;
    uint16_t addr = 0;
    uint32_t statReg;

    int32_t numValues = sizeof(calibrateBuffer) / sizeof(calibrateBuffer[0]);
    // Read measurement values into the calibrate buffer
    if (status == 0)
    {
        status = Admx200xReadFifo(pAdmx200x, &calibrateBuffer[0], &numValues);
    }
    // Populate the fifo with sweep points
    if (status == 0)
    {
        cmdId = CMD_READ_SWEEP_POINTS;
        numValues /= 2;
        status = Admx200xSendCmd(pAdmx200x, &cmdId, &addr,
                                 (uint32_t *)&numValues, &statReg);
    }
    // Read sweep points into the sweep points buffer
    if (status == 0)
    {
        status = Admx200xReadFifo(pAdmx200x, &sweepPoints[0], &numValues);
    }

    if (status == 0)
    {
        for (i = 0; i < numValues; i++)
        {
            DisplayCalibratedValue(calibrateBuffer[2 * i],
                                   calibrateBuffer[2 * i + 1], sweepPoints[i]);
        }
    }

    return status;
}

/**
 * @brief  Displays the output value
 * @param [in] displayVal1 - display value 1
 * @param [in] displayVal2 - display value 2
 * @param [in] sweepVar - sweep variable
 */
void DisplayCalibratedValue(double displayVal1, double displayVal2,
                            double sweepVar)
{
    // Format the measured value and sweep variable
    snprintf(&displayString[0], MAX_CAL_DISPLAY_LENGTH,
             "Count = %d, Value = (%0.6e) + (%0.6e) j\n\r",
             (int32_t)sweepVar, displayVal1, displayVal2);
    // Print the formated string
    printf("%s", &displayString[0]);
}
