/***************************************************************************//**
 *   @file   gpio.cpp
 *   @brief  Implementation of GPIO No-OS platform driver interfaces
********************************************************************************
 * Copyright (c) 2019, 2020 Analog Devices, Inc.
 *
 * All rights reserved.
 *
 * This software is proprietary to Analog Devices, Inc. and its licensors.
 * By using this software you agree to the terms of the associated
 * Analog Devices Software License Agreement.
*******************************************************************************/

/******************************************************************************/
/************************ Includes Files *******************************/
/******************************************************************************/
#include <stdio.h>
#include <mbed.h>

#include "platform_drivers.h"
#include "gpio_extra.h"

/******************************************************************************/
/************************ Functions Definitions *******************************/
/******************************************************************************/

/**
 * @brief Obtain the GPIO decriptor.
 * @param desc - The GPIO descriptor.
 * @param gpio_number - The number of the GPIO.
 * @return ADI_SUCCESS in case of success, ADI_FAILURE otherwise.
 */
int32_t gpio_get(struct gpio_desc **desc, const gpio_init_param *param)
{
	if (desc) {
		// Create the gpio description object for the device
		gpio_desc *new_gpio = (gpio_desc *)malloc(sizeof(gpio_desc));
		if (new_gpio == NULL) {
			return ADI_FAILURE;
		}

		new_gpio->number = param->number;
		*desc = new_gpio;

		return ADI_SUCCESS;
	}

	return ADI_FAILURE;
}


/**
 * @brief Free the resources allocated by gpio_get().
 * @param desc - The GPIO descriptor.
 * @return ADI_SUCCESS in case of success, ADI_FAILURE otherwise.
 */
int32_t gpio_remove(struct gpio_desc *desc)
{
	if (desc) {
		// Free the gpio object
		if (((mbed_gpio_desc *)(desc->extra))->gpio_pin) {
			free(((mbed_gpio_desc *)(desc->extra))->gpio_pin);
		}

		// Free the gpio extra descriptor object
		if ((mbed_gpio_desc *)(desc->extra)) {
			free((mbed_gpio_desc *)(desc->extra));
		}

		// Free the gpio descriptor object
		free(desc);

		return ADI_SUCCESS;
	}

	return ADI_FAILURE;
}


/**
 * @brief Enable the input direction of the specified GPIO.
 * @param desc - The GPIO descriptor.
 * @return ADI_SUCCESS in case of success, ADI_FAILURE otherwise.
 */
int32_t gpio_direction_input(struct gpio_desc *desc)
{
	DigitalIn *gpio_input;  	// pointer to gpio input object
	mbed_gpio_desc *gpio_desc_extra;  // pointer to gpio desc extra parameters

	if (desc) {
		// Configure and instantiate GPIO pin as input
		gpio_input = new DigitalIn((PinName)desc->number);
		if (gpio_input == NULL) {
			return ADI_FAILURE;
		}

		// Create the gpio extra descriptor object to store new gpio instance
		gpio_desc_extra = (mbed_gpio_desc *)malloc(sizeof(mbed_gpio_desc));
		if (gpio_desc_extra == NULL) {
			return ADI_FAILURE;
		}

		gpio_desc_extra->gpio_pin = (mbed_gpio_desc *)gpio_input;
		desc->extra = (mbed_gpio_desc *)gpio_desc_extra;

		// Set the gpio pin mode
		gpio_input->mode((PinMode)((mbed_gpio_init_param *)desc->extra)->pin_mode);

		return ADI_SUCCESS;
	}

	return ADI_FAILURE;
}


/**
 * @brief Enable the output direction of the specified GPIO.
 * @param desc - The GPIO descriptor.
 * @param value - The value.
 *                Example: GPIO_HIGH
 *                         GPIO_LOW
 * @return ADI_SUCCESS in case of success, ADI_FAILURE otherwise.
 */
int32_t gpio_direction_output(struct gpio_desc *desc, uint8_t value)
{
	DigitalOut *gpio_output;   	// pointer to gpio output object
	mbed_gpio_desc *gpio_desc_extra;  // pointer to gpio desc extra parameters

	if(desc) {
		// Configure and instantiate GPIO pin as output
		gpio_output = new DigitalOut((PinName)desc->number);
		if (gpio_output == NULL) {
			return ADI_FAILURE;
		}

		// Create the gpio extra descriptor object to store new gpio instance
		gpio_desc_extra = (mbed_gpio_desc *)malloc(sizeof(mbed_gpio_desc));
		if (gpio_desc_extra == NULL) {
			return ADI_FAILURE;
		}

		gpio_desc_extra->gpio_pin = (mbed_gpio_desc *)gpio_output;
		desc->extra = (mbed_gpio_desc *)gpio_desc_extra;

		return ADI_SUCCESS;
	}

	if (value) {
		// Unused variable - fix compiler warning
	}

	return ADI_FAILURE;
}


/**
 * @brief Get the direction of the specified GPIO.
 * @param desc - The GPIO descriptor.
 * @param direction - The direction.
 *                    Example: GPIO_OUT
 *                             GPIO_IN
 * @return ADI_SUCCESS in case of success, ADI_FAILURE otherwise.
 */
int32_t gpio_get_direction(struct gpio_desc *desc, uint8_t *direction)
{
	if (desc) {
		// Unused variable - fix compiler warning
	}

	if (direction) {
		// Unused variable - fix compiler warning
	}

	return ADI_SUCCESS;
}


/**
 * @brief Set the value of the specified GPIO.
 * @param desc - The GPIO descriptor.
 * @param value - The value.
 *                Example: GPIO_HIGH
 *                         GPIO_LOW
 * @return ADI_SUCCESS in case of success, ADI_FAILURE otherwise.
 */
int32_t gpio_set_value(struct gpio_desc *desc, uint8_t value)
{
	DigitalOut *gpio_output;		// pointer to gpio output object

	if (desc) {
		gpio_output = (DigitalOut *)((mbed_gpio_desc *)desc->extra)->gpio_pin;
		gpio_output->write(value);

		return ADI_SUCCESS;
	}

	if (value) {
		// Unused variable - fix compiler warning
	}

	return ADI_FAILURE;
}


/**
 * @brief Get the value of the specified GPIO.
 * @param desc - The GPIO descriptor.
 * @param value - The value.
 *                Example: GPIO_HIGH
 *                         GPIO_LOW
 * @return ADI_SUCCESS in case of success, ADI_FAILURE otherwise.
 */
int32_t gpio_get_value(struct gpio_desc *desc, uint8_t *value)
{
	DigitalIn *gpio_input;   	// pointer to gpio input object
	uint8_t returnVal = ADI_FAILURE;

	if (desc) {
		gpio_input = (DigitalIn *)((mbed_gpio_desc *)desc->extra)->gpio_pin;
		*value = (uint8_t)gpio_input->read();
		returnVal = gpio_input->is_connected() ? ADI_SUCCESS : ADI_FAILURE;

		return returnVal;
	}

	return ADI_FAILURE;
}
