/*******************************************************************************
    @file:   main.cpp
    
    @brief:  Main module for EVAL-AD910x application interface
--------------------------------------------------------------------------------
    Copyright (c) 2021 Analog Devices, Inc.
    All rights reserved.

    This software is proprietary to Analog Devices, Inc. and its licensors.
    By using this software you agree to the terms of the associated
    Analog Devices Software License Agreement.
    
    20210120-LWSC-CTSLA
*******************************************************************************/

/*** Libraries ***/
#include "mbed.h"
#include "platform/mbed_thread.h"
#include "config.h"
#include "ad910x.h"

/*** Defines for UART Protocol ***/
#define BAUD_RATE       115200

AD910x device( PA_15 );                 
DigitalOut en_cvddx( PG_7, 0 );         ///DigitalOut instance for enable pin of on-board oscillator supply
DigitalOut shdn_n_lt3472( PG_9, 0 );    ///DigitalOut instance for shutdown/enable pin of on-board amplifier supply

/* Configure and instantiate UART protocol and baud rate */
UnbufferedSerial pc( USBTX, USBRX, BAUD_RATE );

/*** Function Declarations ***/
void setup_device( void );
void print_title( void );
void print_prompt1( void );
void print_prompt2( void );
void print_menu( void );
void play_example1( void );
void play_example2( void );
void play_example3( void );
void play_example4( void );
void play_example5( void );
void play_example6( void );
void print_prompt3( void );
void stop_example( void );
void print_prompt4( void );

/*** Main Function ***/

int main()
{
    char ext_clk = 'y';
    char amp_out = 'n';
    char stop = 'n';
    char exit = 'n';
    char example = 1;
    uint8_t connected = 1;
    
    device.resetb = 1;
    device.triggerb = 1;
    setup_device();
    
    print_title();
    
    while( connected == 1 ){
        print_prompt1();
        while( pc.readable() == 0 );
        ext_clk = getchar();
        if ( ext_clk == 'y' ){
            en_cvddx = 0;
            printf("Please connect external clock source to J10.\n");
        } else {
            en_cvddx = 1;
            printf("On-board oscillator supply is enabled.\n");
        }
        thread_sleep_for(500);
        print_prompt2();
        while( pc.readable() == 0 );
        amp_out = getchar();
        if ( amp_out == 'y' ){
            shdn_n_lt3472 = 1;
            printf("On-board amplifier supply is enabled.\n");
        } else {
            shdn_n_lt3472 = 0;
        }
        print_menu();
        example = getchar();
        while( pc.readable() == 0 );
        printf("%c\n", example);
        switch (example) {
            case '1':
                play_example1();
                break;

            case '2':
                play_example2();
                break;
                
            case '3':
                play_example3();
                break;

            case '4':
                play_example4();
                break;
            
            case '5':    
                play_example5();
                break;
                
            case '6':
                play_example6();
                break;
                
            default:
                printf("\n****Invalid Entry****\n\n");
                break;
        }
        print_prompt3();
        while( pc.readable() == 0 );
        while ( !(stop == 'y') ){
            stop = getchar();
        }
        stop_example();
        print_prompt4();
        while( pc.readable() == 0 );
        exit = getchar();
        if ( exit == 'y' ){
            connected = 0;
            printf("\nExiting program...\n");
        }else{
            stop = 'n';
        }
        
    }
    
    return 0;

}

/*** Function Definitions ***/

/*Function to setup SPI communication*/
void setup_device()
{
    device.spi_init( WORD_LEN, POL, FREQ );
    device.AD910x_reg_reset();
}

/*Function to print the title block when program first starts*/
void print_title()
{
    printf("\n***********************************************************************\n");
    printf("* EVAL-%s Demonstration Program                                   *\n", ACTIVE_DEVICE);
    printf("*                                                                     *\n");
    printf("* This program demonstrates how to generate waveforms with the %s *\n", ACTIVE_DEVICE);
    printf("* using example register configurations in the datasheet.             *\n");
    printf("* This program is developed on Mbed OS version 6.                     *\n");
    printf("***********************************************************************\n");   
}

/*Function to print the first prompt/question to user to setup on-board oscillator supply*/
void print_prompt1()
{
    printf( "\nUsing external clock source?\n" );
    printf( "If yes, press y. If no, press any other key.\n" );
}

/*Function to print the next prompt/question to user to setup on-board amplifier supply*/
void print_prompt2()
{
    printf( "\nConnected DAC outputs to on-board amplifiers?\n" );
    printf( "If yes, press y. If no, press any other key.\n" );
}

/*Function to print summary of example AD910x configurations to play*/
void print_menu()
{
    printf("\nExample Summary\n");
    if ( ACTIVE_DEVICE == "AD9106" ){
        printf("   1 - 4 Gaussian Pulses with Different Start Delays and Digital Gain Settings\n");
        printf("   2 - 4 Pulses Generated from an SRAM Vector\n");
        printf("   3 - 4 Pulsed DDS-Generated Sine Waves with Different Start Delays and Digital Gain Settings\n");
        printf("   4 - Pulsed DDS-Generated Sine Wave and 3 Sawtooth Generator Waveforms\n");
        printf("   5 - Pulsed DDS-Generated Sine Waves Amplitude-Modulated by an SRAM Vector\n");
        printf("   6 - DDS-Generated Sine Wave and 3 Sawtooth Waveforms\n");
    }else if( ACTIVE_DEVICE == "AD9102" ){
        printf("   1 - Gaussian Pulse\n");
        printf("   2 - Pulse Generated from an SRAM Vector\n");
        printf("   3 - Pulsed DDS-Generated Sine Wave\n");
        printf("   4 - Sawtooth Waveform\n");
        printf("   5 - Pulsed DDS-Generated Sine Wave Amplitude-Modulated by an SRAM Vector\n");
        printf("   6 - DDS-Generated Sine Wave\n");
    }
    printf("Select an option: ");
}

/*Function to play example 1*/
void play_example1()
{
    if ( ACTIVE_DEVICE == "AD9106" ){
        printf("\n4 Gaussian Pulses with Different Start Delays and Digital Gain Settings\n");
        thread_sleep_for(500);
        device.AD910x_update_sram( example1_RAM_gaussian );
        device.AD910x_update_regs( AD9106_example1_regval );
    }else if( ACTIVE_DEVICE == "AD9102" ){
        printf("\nGaussian Pulse\n");
        thread_sleep_for(500);
        device.AD910x_update_sram( example1_RAM_gaussian );
        device.AD910x_update_regs( AD9102_example1_regval );
    }
    device.AD910x_start_pattern();
}

/*Function to play example 2*/
void play_example2()
{
    if ( ACTIVE_DEVICE == "AD9106" ){
        printf("\n4 Pulses Generated from an SRAM Vector\n");
        thread_sleep_for(500);
        device.AD910x_update_sram( example2_4096_ramp );
        device.AD910x_update_regs( AD9106_example2_regval );
    }else if( ACTIVE_DEVICE == "AD9102" ){
        printf("\nPulse Generated from an SRAM Vector\n");
        thread_sleep_for(500);
        device.AD910x_update_sram( example2_4096_ramp );
        device.AD910x_update_regs( AD9102_example2_regval );
    }
    device.AD910x_start_pattern();
}

/*Function to play example 3*/
void play_example3()
{
    if ( ACTIVE_DEVICE == "AD9106" ){
        printf("\n4 Pulsed DDS-Generated Sine Waves with Different Start Delays and Digital Gain Settings\n");
        thread_sleep_for(500);
        device.AD910x_update_regs( AD9106_example3_regval );
    }else if( ACTIVE_DEVICE == "AD9102" ){
        printf("\nPulsed DDS-Generated Sine Wave\n");
        thread_sleep_for(500);
        device.AD910x_update_regs( AD9102_example3_regval );
    }
    device.AD910x_start_pattern();       
}

/*Function to play example 4*/
void play_example4()
{
    if ( ACTIVE_DEVICE == "AD9106" ){
        printf("\nPulsed DDS-Generated Sine Wave and 3 Sawtooth Generator Waveforms\n");
        thread_sleep_for(500);
        device.AD910x_update_regs( AD9106_example4_regval );
    }else if( ACTIVE_DEVICE == "AD9102" ){
        printf("\nSawtooth Waveform\n");
        thread_sleep_for(500);
        device.AD910x_update_regs( AD9102_example4_regval );
    }
    device.AD910x_start_pattern();       
}

/*Function to play example 5*/
void play_example5()
{
    if ( ACTIVE_DEVICE == "AD9106" ){
        printf("\n4 Pulses Generated from an SRAM Vector\n");
        thread_sleep_for(500);
        device.AD910x_update_sram( example5_RAM_gaussian );
        device.AD910x_update_regs( AD9106_example5_regval );
    }else if ( ACTIVE_DEVICE == "AD9102"){
        printf("\nPulse Generated from an SRAM Vector\n");
        thread_sleep_for(500);
        device.AD910x_update_sram( example5_RAM_gaussian );
        device.AD910x_update_regs( AD9102_example5_regval );
    }
    device.AD910x_start_pattern();   
}

/*Function to play example 1*/
void play_example6()
{
    if ( ACTIVE_DEVICE == "AD9106" ){
        printf("\nDDS-Generated Sine Wave and 3 Sawtooth Waveforms\n");
        thread_sleep_for(500);
        device.AD910x_update_regs( AD9106_example6_regval );
    }else if( ACTIVE_DEVICE == "AD9102" ){
        printf("\nDDS-Generated Sine Wave\n");
        thread_sleep_for(500);
        device.AD910x_update_regs( AD9102_example6_regval );
    }
    device.AD910x_start_pattern();      
}

/*Function to print prompt/question to user to stop pattern*/
void print_prompt3()
{
    printf( "\nStop pattern?\n" );
    printf( "If yes, press y.\n" );
}

/*Function to stop pattern generation*/
void stop_example()
{
    device.AD910x_stop_pattern();
    printf( "Pattern stopped.\n" );
}

/*Function to print prompt/question to user to exit program*/
void print_prompt4()
{
    printf( "\nExit program?\n" );
    printf( "If yes, press y.\n" );
}