/***************************************************************************//**
 *   @file    ad77681_data_capture.c
 *   @brief   Data capture interface for AD7768-1 IIO application
 *   @details This module handles the AD7768-1 data capturing
********************************************************************************
 * Copyright (c) 2021 Analog Devices, Inc.
 *
 * All rights reserved.
 *
 * This software is proprietary to Analog Devices, Inc. and its licensors.
 * By using this software you agree to the terms of the associated
 * Analog Devices Software License Agreement.
*******************************************************************************/

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include <string.h>
#include <stdlib.h>

#include "app_config.h"
#include "ad77681_iio.h"
#include "ad77681_data_capture.h"
#include "adc_data_capture.h"
#include "error.h"

/******************************************************************************/
/********************** Macros and Constants Definition ***********************/
/******************************************************************************/

/* AD77681 ADC Resolution */
#define AD77681_ADC_RESOLUTION			24
/* AD77681 Buffer Length */
#define AD77681_SAMPLE_DATA_BUFF_LEN	6
/* AD77681 24 Bits Sign Extension Value */
#define AD77681_24_BITS_SIGN_EXTENSION	0xFFFFFF
/* AD77681 2 Bytes Shift Value */
#define	AD77681_2_BYTES_SHIFT			16
/* AD77681 1 Byte Shift Value */
#define	AD77681_1_BYTE_SHIFT			8

/******************************************************************************/
/********************** Variables and User Defined Data Types *****************/
/******************************************************************************/

/******************************************************************************/
/************************ Functions Declarations ******************************/
/******************************************************************************/

/* Perform single sample read pre-operations */
static int32_t ad77681_single_sample_read_start_ops(uint8_t input_chn);
/* Perform continuous sample read pre-operations */
static int32_t ad77681_continuous_sample_read_start_ops(uint32_t chn_mask);
/* Perform conversion on input channel and read single conversion sample */
static int32_t ad77681_perform_conv_and_read_sample(uint32_t *read_adc_data,
		uint8_t chn);
/* Read ADC sampled/conversion data */
static int32_t ad77681_read_converted_sample(uint32_t *adc_raw, uint8_t chn_indx);

/* Define the variable for data_capture_ops structure */
struct data_capture_ops data_capture_ops = {
	/* Point AD77681 data capture functions to generic ADC data capture functions */
	.single_sample_read_start_ops = ad77681_single_sample_read_start_ops,	
	.perform_conv_and_read_sample = ad77681_perform_conv_and_read_sample,	
	.single_sample_read_stop_ops = NULL,
	.continuous_sample_read_start_ops = ad77681_continuous_sample_read_start_ops,
	.read_converted_sample = ad77681_read_converted_sample,
	.continuous_sample_read_stop_ops = NULL,
	.trigger_next_conversion = NULL,
};

/******************************************************************************/
/************************ Functions Definitions *******************************/
/******************************************************************************/

/*!
 * @brief	Perform the operations required before starting continuous sample read
 * @param	input_chn[out] - ADC input channel
 * @return	SUCCESS in case of success, FAILURE otherwise
 */
int32_t ad77681_single_sample_read_start_ops(uint8_t input_chn)
{

	int32_t single_read_sts = FAILURE;

	single_read_sts = ad77681_set_conv_mode
			  (
				  p_ad77681_dev_inst,
				  AD77681_CONV_SINGLE,
				  AD77681_AIN_SHORT,
				  false
			  );

	return single_read_sts;
}

/*!
 * @brief	Perform the operations required before starting continuous sample read
 * @param	chn_mask[in] - holds the list of all active channels set from IIO client for data capturing
 * @return	SUCCESS in case of success, FAILURE otherwise
 */
int32_t ad77681_continuous_sample_read_start_ops(uint32_t chn_mask)
{
	int32_t cont_read_sts = FAILURE;

	cont_read_sts = ad77681_set_conv_mode
			(
				p_ad77681_dev_inst,
				AD77681_CONV_CONTINUOUS,
				AD77681_AIN_SHORT,
				false
			);

	return cont_read_sts;
}

/*!
 * @brief	Read ADC raw data for recently sampled channel
 * @param	adc_raw[out] - Pointer to adc data read variable
 * @param	chn_indx[in] - Channel for which data is to read
 * @return	SUCCESS in case of success, FAILURE otherwise
 * @note	This function is intended to call from the conversion end trigger
 *			event. Therefore, this function should just read raw ADC data
 *			without further monitoring conversion end event
 */
int32_t ad77681_read_converted_sample(uint32_t *adc_raw, uint8_t chn_indx)
{
	uint32_t	ad77681_sample_data = 0;
	uint8_t		ad77681_sample_data_buff[AD77681_SAMPLE_DATA_BUFF_LEN] = { 0, 0, 0, 0, 0, 0 };

	if (!adc_raw) {
		return FAILURE;
	}

	if (ad77681_spi_read_adc_data(p_ad77681_dev_inst, ad77681_sample_data_buff,
				      AD77681_REGISTER_DATA_READ) != SUCCESS) {
		return FAILURE;
	}

	ad77681_sample_data = (uint32_t)
			      (
				      (ad77681_sample_data_buff[1] << AD77681_2_BYTES_SHIFT) |
				      (ad77681_sample_data_buff[2] << AD77681_1_BYTE_SHIFT ) |
				      (ad77681_sample_data_buff[3])
			      );
	*adc_raw = ad77681_sample_data & AD77681_24_BITS_SIGN_EXTENSION;

	return SUCCESS;
}

/*!
 * @brief	Read ADC single sample data
 * @param	read_adc_data[out] - Pointer to adc data read variable
 * @param	chn[in] - Channel for which data is to read
 * @return	SUCCESS in case of success, FAILURE otherwise
 * @details	This function performs the sampling on previously active channel
 *			and then read conversion result
 */
int32_t ad77681_perform_conv_and_read_sample(uint32_t *read_adc_data,
		uint8_t chn)
{
	uint32_t ad77681_sample_data = 0;

	if (ad77681_read_converted_sample(&ad77681_sample_data, 0) != SUCCESS) {
		return FAILURE;
	}
	*read_adc_data = ad77681_sample_data;

	return SUCCESS;
}
