/***************************************************************************//**
 *   @file    app_config.c
 *   @brief   Application configurations module
 *   @details This module contains the configurations needed for IIO application
********************************************************************************
 * Copyright (c) 2021 Analog Devices, Inc.
 * All rights reserved.
 *
 * This software is proprietary to Analog Devices, Inc. and its licensors.
 * By using this software you agree to the terms of the associated
 * Analog Devices Software License Agreement.
*******************************************************************************/

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include <stdbool.h>

#include "app_config.h"
#include "adc_data_capture.h"
#include "ad77681_iio.h"
#include "error.h"
#include "uart.h"
#include "irq.h"
#include "gpio.h"

/******************************************************************************/
/************************ Macros/Constants ************************************/
/******************************************************************************/

/******************************************************************************/
/*************************** Types Declarations *******************************/
/******************************************************************************/

/*
 * UART init parameters structure
 */
struct uart_init_param uart_init_params = {
	.device_id = NULL,
	.baud_rate = IIO_UART_BAUD_RATE,
	.extra = &uart_init_extra_params
};

/* LED GPIO descriptor */
struct gpio_init_param gpio_init_led_green = {
	.number = LED_GREEN,
	.extra = NULL
};

/* External interrupt init parameters */
struct irq_init_param external_int_init_param = {
	.irq_ctrl_id = EXTERNAL_INT_ID1,
	.extra = &mbed_ext_int_init_param
};

/* External interrupt callback descriptor */
struct callback_desc external_int_callback_desc = {
	data_capture_callback,
	NULL,
	NULL
};

/* LED GPIO descriptor */
gpio_desc *led_green_gpio_desc = NULL;

/* External interrupt descriptor */
struct irq_ctrl_desc *external_int_desc;

/******************************************************************************/
/************************ Functions Prototypes ********************************/
/******************************************************************************/

/******************************************************************************/
/************************ Functions Definitions *******************************/
/******************************************************************************/

/**
 * @brief 	Initialize the GPIOs
 * @return	SUCCESS in case of success, FAILURE otherwise
 * @details	This function initialize the GPIOs used by application
 */
static int32_t init_gpio(void)
{
	/* Initialize the LED GREEN GPIO */
	if (gpio_get_optional(&led_green_gpio_desc, &gpio_init_led_green) != SUCCESS) {
		return FAILURE;
	}

	if (led_green_gpio_desc) {
		if (gpio_direction_output(led_green_gpio_desc, GPIO_HIGH) != SUCCESS) {
			return FAILURE;
		}
	}

	return SUCCESS;
}

/**
 * @brief 	Initialize the IRQ contoller
 * @return	SUCCESS in case of success, FAILURE otherwise
 * @details	This function initialize the interrupts for system peripherals
 */
static int32_t init_interrupt(void)
{
	/* Init interrupt controller for external interrupt */
	if (irq_ctrl_init(&external_int_desc, &external_int_init_param) == FAILURE) {
		return FAILURE;
	}

	/* Register a callback function for external interrupt */
	if (irq_register_callback(external_int_desc,
				  EXTERNAL_INT_ID1,
				  &external_int_callback_desc) == FAILURE) {
		return FAILURE;
	}

	return SUCCESS;
}

/**
 * @brief 	Initialize the system peripherals
 * @return	SUCCESS in case of success, FAILURE otherwise
 */
int32_t init_system(void)
{
	if (init_gpio() != SUCCESS) {
		return FAILURE;
	}

	if (init_interrupt() != SUCCESS) {
		return FAILURE;
	}

	return SUCCESS;
}
