/***************************************************************************//**
 *   @file    app_config.c
 *   @brief   Application configurations module (platform-agnostic)
 *   @details This module performs the system configurations
********************************************************************************
 * Copyright (c) 2021-22 Analog Devices, Inc.
 * All rights reserved.
 *
 * This software is proprietary to Analog Devices, Inc. and its licensors.
 * By using this software you agree to the terms of the associated
 * Analog Devices Software License Agreement.
*******************************************************************************/

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include <stdbool.h>

#include "app_config.h"
#include "ad7689_data_capture.h"
#include "no_os_error.h"
#include "no_os_uart.h"
#include "no_os_gpio.h"
#include "no_os_irq.h"
#include "no_os_pwm.h"

/******************************************************************************/
/************************ Macros/Constants ************************************/
/******************************************************************************/

/******************************************************************************/
/******************** Variables and User Defined Data Types *******************/
/******************************************************************************/

/* UART init parameters */
static struct no_os_uart_init_param uart_init_params = {
	.device_id = NULL,
	.baud_rate = IIO_UART_BAUD_RATE,
	.size = NO_OS_UART_CS_8,
	.parity = NO_OS_UART_PAR_NO,
	.stop = NO_OS_UART_STOP_1_BIT,
	.extra = &uart_extra_init_params
};

/* LED GPO init parameters */
static struct no_os_gpio_init_param led_gpio_init_params = {
	.number = LED_GPO,
	.platform_ops = &gpio_ops,
	.extra = NULL
};

/* External interrupt init parameters */
static struct no_os_irq_init_param ext_int_init_params = {
	.irq_ctrl_id = 0,
	.platform_ops = &irq_ops,
	.extra = &ext_int_extra_init_params
};

/* External interrupt callback descriptor */
static struct no_os_callback_desc ext_int_callback_desc = {
	data_capture_callback,
	NULL,
	NULL
};

/* PWM init parameters */
static struct no_os_pwm_init_param pwm_init_params = {
	.id = 0,
	.period_ns = CONV_TRIGGER_PERIOD_NSEC,			// PWM period in nsec
	.duty_cycle_ns = CONV_TRIGGER_DUTY_CYCLE_NSEC,	// PWM duty cycle in nsec
	.extra = &pwm_extra_init_params
};

/* LED GPO descriptor */
struct no_os_gpio_desc *led_gpio_desc;

/* UART descriptor */
struct no_os_uart_desc *uart_desc;

/* External interrupt descriptor */
struct no_os_irq_ctrl_desc *ext_int_desc;

/* PWM descriptor */
struct no_os_pwm_desc *pwm_desc;

/******************************************************************************/
/************************** Functions Declarations ****************************/
/******************************************************************************/

/******************************************************************************/
/************************** Functions Definitions *****************************/
/******************************************************************************/

/**
 * @brief 	Initialize the GPIOs
 * @return	0 in case of success, negative error code otherwise
 */
static int32_t init_gpio(void)
{
	int32_t ret;

	/* Initialize the LED GPO */
	ret = no_os_gpio_get_optional(&led_gpio_desc, &led_gpio_init_params);
	if (ret) {
		return ret;
	}

	ret = no_os_gpio_direction_output(led_gpio_desc, NO_OS_GPIO_HIGH);
	if (ret) {
		return ret;
	}

	return 0;
}

/**
 * @brief 	Initialize the UART peripheral
 * @return	0 in case of success, negative error code otherwise
 */
static int32_t init_uart(void)
{
	return no_os_uart_init(&uart_desc, &uart_init_params);
}

/**
 * @brief 	Initialize the IRQ contoller
 * @return	0 in case of success, negative error code otherwise
 */
static int32_t init_interrupt(void)
{
	int32_t ret;

	/* Init interrupt controller for external interrupt */
	ret = no_os_irq_ctrl_init(&ext_int_desc, &ext_int_init_params);
	if (ret) {
		return ret;
	}

	/* Register a callback function for external interrupt */
	ret = no_os_irq_register_callback(ext_int_desc,
					  EXT_INT_ID,
					  &ext_int_callback_desc);
	if (ret) {
		return ret;
	}

	/* Enable external interrupt */
	ret = no_os_irq_enable(ext_int_desc, EXT_INT_ID);
	if (ret) {
		return ret;
	}

	return 0;
}

/**
 * @brief 	Initialize the PWM contoller
 * @return	0 in case of success, negative error code otherwise
 */
static int32_t init_pwm(void)
{
	int32_t ret;

	/* Initialize the PWM interface to generate PWM signal
	 * on conversion trigger event pin */
	ret = no_os_pwm_init(&pwm_desc, &pwm_init_params);
	if (ret) {
		return ret;
	}

	ret = no_os_pwm_enable(pwm_desc);
	if (ret) {
		return ret;
	}

	return 0;
}

/**
 * @brief 	Initialize the system peripherals
 * @return	0 in case of success, negative error code otherwise
 */
int32_t init_system(void)
{
	int32_t ret;

	ret = init_gpio();
	if (ret) {
		return ret;
	}

	ret = init_uart();
	if (ret) {
		return ret;
	}

#if (DATA_CAPTURE_MODE == CONTINUOUS_DATA_CAPTURE)
	ret = init_interrupt();
	if (ret) {
		return ret;
	}

	ret = init_pwm();
	if (ret) {
		return ret;
	}
#endif

#if defined(USE_SDRAM_CAPTURE_BUFFER)
	ret = sdram_init();
	if (ret) {
		return ret;
	}
#endif

	return 0;
}
