/*************************************************************************//**
 *   @file   ad4696_support.c
 *   @brief  AD469x device No-OS driver supports
******************************************************************************
* Copyright (c) 2021 Analog Devices, Inc.
*
* All rights reserved.
*
* This software is proprietary to Analog Devices, Inc. and its licensors.
* By using this software you agree to the terms of the associated
* Analog Devices Software License Agreement.
*****************************************************************************/

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include <stdint.h>

#include "app_config.h"
#include "ad4696_support.h"
#include "error.h"

/******************************************************************************/
/********************** Macros and Constants Definition ***********************/
/******************************************************************************/

/******************************************************************************/
/********************** Variables and User Defined Data Types *****************/
/******************************************************************************/

/******************************************************************************/
/************************ Functions Definitions *******************************/
/******************************************************************************/
/*!
 * @brief	Enter Manual Trigger mode
 * @param	device[in] - device instance
 * @return	SUCCESS in case of success, negative error code otherwise.
 */
int32_t ad4696_enable_manual_trigger_mode(struct ad469x_dev *device)
{
	do 
	{
		if (ad469x_spi_reg_write(device,
			AD469x_REG_SETUP, 
			AD469x_REG_SETUP_RESET) != SUCCESS) {
			break;
		}
	
		if (ad469x_spi_reg_write(device,
			AD469x_REG_SEQ_CTRL, 
			AD469x_REG_SEQ_CTRL_RESET) != SUCCESS) {
			break;
		}

		return SUCCESS;
	} while (0);
	
	return FAILURE;
}

/*!
 * @brief	Select between polarity modes
 * @param	device[in] - device instance
 * @return	SUCCESS in case of success, negative error code otherwise.
 */
int32_t ad4696_polarity_mode_select(struct ad469x_dev *device)
{
	uint8_t data = 0;

	for (uint8_t i = 0; i < AD469x_CHANNEL_NO;i++)
	{
		if (ad469x_spi_reg_read(device,
			AD469x_REG_CONFIG_IN(i), 
			&data) != SUCCESS) {
			return FAILURE;
		}
		
#if defined(PSEUDO_BIPOLAR_MODE)		 
		data |= (AD469x_REG_CONFIG_IN_PAIR(AD469x_PSEUDO_BIPOLAR_MODE)
				| AD469x_REG_CONFIG_IN_MODE(AD469x_INx_COM));
#else
		data |= (AD469x_REG_CONFIG_IN_PAIR(AD469x_UNIPOLAR_MODE)
				| AD469x_REG_CONFIG_IN_MODE(AD469x_INx_REF_GND));
#endif
		
		if (ad469x_spi_reg_write(device,
			AD469x_REG_CONFIG_IN(i), 
			data) != SUCCESS) {
			return FAILURE;
		}
	}
	return SUCCESS;
}
