/***************************************************************************//*
 * @file    ad4110_temperature_sensor.cpp
 * @brief   AD4110 temperature sensor functionality
 * @details
******************************************************************************
 * Copyright (c) 2022 Analog Devices, Inc. All Rights Reserved.
 *
 * This software is proprietary to Analog Devices, Inc. and its licensors.
 * By using this software you agree to the terms of the associated
 * Analog Devices Software License Agreement.
******************************************************************************/

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include <stdint.h>
#include <ptxxx.h>
#include <thermocouple.h>
#include "ad4110_temperature_sensor.h"

#ifdef __cplusplus
extern "C"
{
#endif //  _cplusplus

#include "app_config.h"
#include "ad4110_support.h"

#ifdef __cplusplus  // Closing extern c
}
#endif //  _cplusplus

/******************************************************************************/
/********************* Macros and Constants Definitions ***********************/
/******************************************************************************/

/* NTC thermistor Rsense value (in ohms) */
#define NTC_RSENSE			10000U

/* RTD Rref Resistance value (in ohms) */
#define RTD_RREF			25000U

/* Lead resistance value (in ohms)
* NOTE: There is an RC filter configuration physically in series with the AIN+
* and AIN- terminals, (10 + 10 ohms) that needs to be compensated for during conversion. */
#define LEAD_RESISTANCE			20

/* Scale factor for TMP36 used for CJC */
#define TMP36_SCALE			0.03

/******************************************************************************/
/******************** Variables and User Defined Data Types *******************/
/******************************************************************************/

/******************************************************************************/
/************************** Functions Definitions *****************************/
/******************************************************************************/

/**
 * @brief Convert ADC raw value into equivalent RTD temperature
 * @param rtd_sample[in] - Raw ADC sample for RTD sensor
 * @return RTD temperature
 * @note Fixed PT100 RTD sensor is used
 */
float get_rtd_temperature(uint32_t rtd_sample)
{
	PT100 rtd_sensor;
	float rtd_resistance;

	rtd_resistance = convert_adc_raw_into_rtd_resistance(rtd_sample, RTD_RREF);
	/* The lead resistance value is compensated to achieve accuracy in resistance and temperature
	measurement */
	rtd_resistance = rtd_resistance - LEAD_RESISTANCE;
	return rtd_sensor.convertResistanceToTemperature(rtd_resistance);
}


/**
 * @brief Convert ADC raw value into TC temperature
 * @param tc_sample[in] - Raw TC sample
 * @param tc_channel[in] - Thermocouple channel
 * @param cjc_sample[in] - Raw CJC sample
 * @param cjc_channel[in - CJC channel
 * @param cjc_temp[in, out] - CJC temperature value
 * @return TC temperature
 * @note T type thermocouple used, on board TMP36 is used for CJC.
 */
float get_tc_temperature(uint32_t tc_sample, uint8_t tc_channel,
			 uint32_t cjc_sample, uint8_t cjc_channel,
			 float *cjc_temp)
{
	Thermocouple_Type_T tcSensor;
	float tc_mv;
	volatile float cjc_mv;
	float tc_temperature;
	float cjc_temperature;

	/* Thermocouple temperature calculation */
	tc_mv = convert_adc_sample_into_voltage(tc_sample, tc_channel) * 1000;
	tc_temperature = tcSensor.convert(tc_mv);

	/* CJC Temperature calculation */
	cjc_mv = convert_adc_sample_into_voltage(cjc_sample, cjc_channel) * 1000;
	cjc_temperature  = TMP36_SCALE * cjc_mv;

	/* Get the CJC temperature */
	*cjc_temp = cjc_temperature;

	return (tc_temperature + cjc_temperature);
}