/*************************************************************************//**
 *   @file   app_config.h
 *   @brief  Configuration file for AD4110_1 IIO firmware application
******************************************************************************
* Copyright (c) 2022 Analog Devices, Inc.
*
* All rights reserved.
*
* This software is proprietary to Analog Devices, Inc. and its licensors.
* By using this software you agree to the terms of the associated
* Analog Devices Software License Agreement.
*****************************************************************************/

#ifndef APP_CONFIG_H
#define APP_CONFIG_H

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include <stdint.h>
#include "no_os_gpio.h"

/******************************************************************************/
/********************** Macros and Constants Definition ***********************/
/******************************************************************************/

/* List of supported platforms*/
#define	MBED_PLATFORM		1

/* List of demo mode configurations */
#define VOLTAGE_MODE_CONFIG	1
#define CURRENT_MODE_CONFIG	2
#define FIELD_POWER_SUPPLY_CONFIG	3
#define THERMOCOUPLE_CONFIG	4
#define RTD_CONFIG		5

#define ACTIVE_DEVICE		"ad4110"

/* Select the active platform */
#if !defined(ACTIVE_PLATFORM)
#define ACTIVE_PLATFORM	    MBED_PLATFORM
#endif

/* Select the demo mode configuration default is RTD mode */
#if !defined(ACTIVE_DEMO_MODE_CONFIG)
#define ACTIVE_DEMO_MODE_CONFIG		RTD_CONFIG
#endif

/* List of data capture modes for AD717x device */
#define CONTINUOUS_DATA_CAPTURE		0
#define BURST_DATA_CAPTURE			1

/* Macros for stringification */
#define XSTR(s)		#s
#define STR(s)		XSTR(s)

#if (ACTIVE_PLATFORM == MBED_PLATFORM)
#include "app_config_mbed.h"

/* Redefine the init params structure mapping w.r.t. platform */
#define uart_extra_init_params mbed_uart_extra_init_params
#define spi_extra_init_params mbed_spi_extra_init_params
#define ext_int_extra_init_params mbed_ext_int_extra_init_params
#define IRQ_INT_ID	EXTERNAL_INT_ID1
#define csb_platform_ops mbed_gpio_ops
#define irq_platform_ops mbed_irq_ops
#define spi_platform_ops mbed_spi_ops
#define HW_CARRIER_NAME		STR(TARGET_NAME)
#else
#error "No/Invalid active platform selected"
#endif

#define HW_MEZZANINE_NAME	"EVAL-AD4110-1SDZ"
#define HW_NAME			ACTIVE_DEVICE

/* Baud rate for IIO application UART interface */
#define IIO_UART_BAUD_RATE	230400

/* Enable the UART/VirtualCOM port connection (default VCOM) */
//#define USE_PHY_COM_PORT		// Uncomment to select UART

#if !defined(USE_PHY_COM_PORT)

/* VCOM Serial number definition */
#define	FIRMWARE_NAME	"ad4110-1_iio-application"

#if !defined(PLATFORM_NAME)
#define PLATFORM_NAME	"SDP_K1"
#endif

#if !defined(EVB_INTERFACE)
#define EVB_INTERFACE	"ARDUINO"
#endif

/* Below USB configurations (VID and PID) are owned and assigned by ADI.
 * If intended to distribute software further, use the VID and PID owned by your
 * organization */
#define VIRTUAL_COM_PORT_VID	0x0456
#define VIRTUAL_COM_PORT_PID	0xb66c

#define VIRTUAL_COM_SERIAL_NUM	(FIRMWARE_NAME "_" PLATFORM_NAME "_" EVB_INTERFACE)
#endif // !defined(USE_PHY_COM_PORT)

/* Select the ADC data capture mode (default is CC mode) */
#if !defined(DATA_CAPTURE_MODE)
#define DATA_CAPTURE_MODE	BURST_DATA_CAPTURE
#endif

/* The below macro defines the register set to be accessed during the debug
 * register read/write via IIO Client. Assign the below macro to AD4110_AFE
 * for AFE register read/write */
#define REGISTER_READ_SEL   A4110_ADC

/* Enable/Disable the use of SDRAM for ADC data capture buffer */
//#define USE_SDRAM_CAPTURE_BUFFER	// Uncomment to use SDRAM as data buffer

/* AD4110 Sampling Rate of the device in SPS, excluding the fractional part.
 * The following are the possible values of sampling frequencies (in SPS):
 *
 * 125000 , 62500, 31250, 25000, 15625, 10390, 4994, 2498,
 * 1000, 500, 395, 200, 100, 59, 49, 20, 16, 10, 5. */

/* Sampling rate- The default sampling rate is 125000 SPS, which is the maximum
 * output data rate. */
#define AD4110_SAMPLING_RATE 125000

/* Assign the ODR bitfield value depending on chosen sampling frequency */
#if (AD4110_SAMPLING_RATE == 125000)
#define AD4110_ODR_SEL	    0
#elif (AD4110_SAMPLING_RATE == 62500)
#define Ad4110_ODR_SEL	    2
#elif (AD4110_SAMPLING_RATE == 31250)
#define AD4110_ODR_SEL	    4
#elif (AD4110_SAMPLING_RATE == 25000)
#define AD4110_ODR_SEL	    5
#elif (AD4110_SAMPLING_RATE == 15625)
#define AD4110_ODR_SEL	    6
#elif (AD4110_SAMPLING_RATE == 10390)
#define AD4110_ODR_SEL	    7
#elif (AD4110_SAMPLING_RATE == 4994)
#define AD4110_ODR_SEL	    8
#elif (AD4110_SAMPLING_RATE == 2498)
#define AD4110_ODR_SEL	    9
#elif (AD4110_SAMPLING_RATE == 1000)
#define AD4110_ODR_SEL	    10
#elif (AD4110_SAMPLING_RATE == 500)
#define AD4110_ODR_SEL	    11
#elif (AD4110_SAMPLING_RATE == 395)
#define AD4110_ODR_SEL	    12
#elif (AD4110_SAMPLING_RATE == 200)
#define AD4110_ODR_SEL	    13
#elif (AD4110_SAMPLING_RATE == 100)
#define AD4110_ODR_SEL	    14
#elif (AD4110_SAMPLING_RATE == 59)
#define AD4110_ODR_SEL	    15
#elif (AD4110_SAMPLING_RATE == 49)
#define AD4110_ODR_SEL	    16
#elif (AD4110_SAMPLING_RATE == 20)
#define AD4110_ODR_SEL	    17
#elif (AD4110_SAMPLING_RATE == 16)
#define AD4110_ODR_SEL	    18
#elif (AD4110_SAMPLING_RATE == 10)
#define AD4110_ODR_SEL	    19
#elif (AD4110_SAMPLING_RATE == 5)
#define AD4110_ODR_SEL	    20
#else
#warning "Invalid sampling frequency selection, using 125000 as default"
#define AD4110_SAMPLING_RATE 125000
#define AD4110_ODR_SEL	     0
#endif

/* Include user config files and params according to active/selected demo mode config */
#if (ACTIVE_DEMO_MODE_CONFIG == VOLTAGE_MODE_CONFIG)
#include "ad4110_user_config.h"
#define ad4110_init_params	ad4110_user_config_params
#elif (ACTIVE_DEMO_MODE_CONFIG == CURRENT_MODE_CONFIG)
#include "ad4110_current_mode_config.h"
#define ad4110_init_params	ad4110_current_mode_init_params
#elif (ACTIVE_DEMO_MODE_CONFIG == FIELD_POWER_SUPPLY_CONFIG)
#include "ad4110_field_power_supply_config.h"
#define ad4110_init_params	ad4110_field_power_supply_mode_init_params
#elif (ACTIVE_DEMO_MODE_CONFIG == THERMOCOUPLE_CONFIG)
#include "ad4110_thermocouple_mode_config.h"
#define ad4110_init_params	ad4110_thermocouple_mode_init_params
#elif (ACTIVE_DEMO_MODE_CONFIG == RTD_CONFIG)
#include "ad4110_rtd_mode_config.h"
#define ad4110_init_params ad4110_rtd_mode_init_params
#endif

/* ADC Resolution */
#define AD4110_RESOLUTION   24

/* ADC internal reference voltage */
#define AD4110_REF_VOLTAGE  2.5

/* ADC number of channels */
#define AD4110_NUM_CHANNELS 4

/* ADC max count (full scale value) for unipolar inputs */
#define ADC_MAX_COUNT_UNIPOLAR	(uint32_t)((1 << AD4110_RESOLUTION) - 1)

/* ADC max count (full scale value) for bipolar inputs */
#define ADC_MAX_COUNT_BIPOLAR	(uint32_t)(1 << (AD4110_RESOLUTION-1))

/******************************************************************************/
/********************** Variables and User Defined Data Types *****************/
/******************************************************************************/

/******************************************************************************/
/************************ Public Declarations *********************************/
/******************************************************************************/

extern struct no_os_uart_desc *uart_desc;
extern no_os_gpio_desc *csb_gpio;
extern struct no_os_irq_ctrl_desc *external_int_desc;
int32_t init_system(void);

#endif // APP_CONFIG_H

