/***************************************************************************//**
 *   @file    app_config.c
 *   @brief   Application configurations module
********************************************************************************
 * Copyright (c) 2022 Analog Devices, Inc.
 * All rights reserved.
 *
 * This software is proprietary to Analog Devices, Inc. and its licensors.
 * By using this software you agree to the terms of the associated
 * Analog Devices Software License Agreement.
*******************************************************************************/

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include "app_config.h"
#include "no_os_uart.h"
#include "ad4110_data_capture.h"

/******************************************************************************/
/************************ Macros/Constants ************************************/
/******************************************************************************/

/******************************************************************************/
/*************************** Types Declarations *******************************/
/******************************************************************************/

/* UART init parameters */
struct no_os_uart_init_param uart_init_params = {
	.device_id = NULL,
	.baud_rate = IIO_UART_BAUD_RATE,
	.size = NO_OS_UART_CS_8,
	.parity = NO_OS_UART_PAR_NO,
	.stop = NO_OS_UART_STOP_1_BIT,
	.extra = &uart_extra_init_params
};

/* GPIO - Chip select Pin init parameters */
static struct no_os_gpio_init_param csb_init_param = {
	.number = SPI_CSB,
	.platform_ops = &csb_platform_ops
};

#if (DATA_CAPTURE_MODE == CONTINUOUS_DATA_CAPTURE)
/* External interrupt init parameters */
static struct no_os_irq_init_param ext_int_init_params = {
	.irq_ctrl_id = 0,
	.extra = &ext_int_extra_init_params,
	.platform_ops = &irq_platform_ops,
};
/* External interrupt callback descriptor */
static struct no_os_callback_desc ext_int_callback_desc = {
	data_capture_callback,
	NULL,
	NULL
};
#endif

/* UART descriptor */
struct no_os_uart_desc *uart_desc;

/* GPIO descriptor for the chip select pin */
no_os_gpio_desc *csb_gpio;

/* External interrupt descriptor */
struct no_os_irq_ctrl_desc *external_int_desc;

/******************************************************************************/
/************************ Functions Prototypes ********************************/
/******************************************************************************/

/******************************************************************************/
/************************ Functions Definitions *******************************/
/******************************************************************************/

/**
 * @brief Initialize the UART peripheral
 * @return 0 in case of success, negative error code otherwise
 */
static int32_t init_uart(void)
{
	return no_os_uart_init(&uart_desc, &uart_init_params);
}


#if (DATA_CAPTURE_MODE == CONTINUOUS_DATA_CAPTURE)
/**
 * @brief Initialize the IRQ contoller
 * @return 0 in case of success, negative error code otherwise
 * @details This function initialize the interrupts for system peripherals
 */
int32_t init_interrupt(void)
{
	int32_t ret;

	do {
		/* Init interrupt controller for external interrupt */
		ret = no_os_irq_ctrl_init(&external_int_desc, &ext_int_init_params);
		if (ret) {
			break;
		}

		/* Register a callback function for external interrupt */
		ret = no_os_irq_register_callback(external_int_desc,
						  IRQ_INT_ID,
						  &ext_int_callback_desc);
		if (ret) {
			break;
		}

		return 0;
	} while (0);

	return ret;
}
#endif


/**
 * @brief Initialize the system peripherals
 * @return 0 in case of success, Negative error code otherwise
 */
int32_t init_system(void)
{
	int32_t ret;

	ret = init_uart();
	if (ret) {
		return ret;
	}

#if defined(USE_SDRAM_CAPTURE_BUFFER)
	ret = sdram_init();
	if (ret) {
		return ret;
	}
#endif

#if (DATA_CAPTURE_MODE == CONTINUOUS_DATA_CAPTURE)
	ret = init_interrupt();
	if (ret) {
		return ret;
	}

	ret = no_os_gpio_get(&csb_gpio, &csb_init_param);
	if (ret) {
		return ret;
	}

	ret = no_os_gpio_direction_output(csb_gpio, NO_OS_GPIO_HIGH);
	if (ret) {
		return ret;
	}
#endif

	return 0;
}