# @file    ad4110_sensor_measurement.py
# @brief   Sensor Measurement steps for the demo  mode configurations in AD4110
# 
# Copyright (c) 2022 Analog Devices, Inc.
# All rights reserved.
# 
# This software is proprietary to Analog Devices, Inc. and its licensors.
# By using this software you agree to the terms of the associated
# Analog Devices Software License Agreement.
#

import serial
from time import sleep
from pynput import keyboard
from adi.ad4110 import *
from ad4110_xattr import *

# Delays in seconds
short_time = 0.5
long_time = 1

def key_press_event(key):
    global key_pressed
    key_pressed = True

def init_sensor_measurement():
    global device
    global demo_config
    global listener

    ######## User configuration ##########
    # Configure the backend for PC to IIOD interface
    uri = "serial:COM17,230400"  # For UART, baud rate must be same as set in the FW. COM port is physical Or VCOM.
    device_name = "ad4110"      # Name of the device must be same as set in the FW.
    ######################################

    # Create an IIO device context
    device = ad4110_xattr(uri, device_name)
    device._ctx.set_timeout(100000)
    device.rx_output_type = "raw"

    # Get current user device config from the firmware
    demo_config = device.demo_config
    print("\r\nDemo Config: {}\r\n".format(demo_config))

    listener = keyboard.Listener(on_press=key_press_event)
    listener.start()

def perform_sensor_measurement():
    global key_pressed 

    print("\r\n*** Press any key to stop the measurement ***\r\n")
    sleep(long_time)
    
    # Print the header
    header = ""
    for chn in device.channel:
        header = header + chn.name + ' '
    print(header)
    
    key_pressed = False
    while not key_pressed:
        result_str = ""
        for chn in device.channel:
            sleep(short_time)
            adc_raw = chn.raw
            adc_scale = chn.scale
            adc_offset = chn.offset
            if (demo_config == 'Voltage'):
                actual_voltage = (adc_raw + adc_offset) * adc_scale / 1000
                result_str = result_str + str(round(actual_voltage,4)) + 'V  '
            elif ((demo_config == 'Current') or (demo_config == 'Field Power Supply')):
                actual_current = (adc_raw + adc_offset) * adc_scale
                result_str = result_str + str(round(actual_current,4)) + 'mA  '
            elif (demo_config == 'RTD') or (demo_config == 'Thermocouple'):
                actual_temperature = (adc_raw * adc_scale) / 1000
                result_str = result_str + str(round(actual_temperature,4)) + 'C  '
        print(result_str)

def exit():
    global listener
    global device

    # Delete the objects
    del listener
    del device

def main():
    init_sensor_measurement()
    perform_sensor_measurement()
    exit()

if __name__ == "__main__":
    main()