/***************************************************************************//*
 * @file    ad4110_support.c
 * @brief   AD4110 No-OS driver support file
******************************************************************************
 * Copyright (c) 2022 Analog Devices, Inc. All Rights Reserved.
 *
 * This software is proprietary to Analog Devices, Inc. and its licensors.
 * By using this software you agree to the terms of the associated
 * Analog Devices Software License Agreement.
******************************************************************************/

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/

#include "app_config.h"
#include "ad4110_iio.h"

/******************************************************************************/
/********************* Macros and Constants Definitions ***********************/
/******************************************************************************/

#define AD4110_LV_CHANNEL_PGA	    1

/******************************************************************************/
/******************** Variables and User Defined Data Types *******************/
/******************************************************************************/

/******************************************************************************/
/************************** Functions Definitions *****************************/
/******************************************************************************/

/*!
 * @brief Perform sign conversion for handling negative voltages in bipolar mode
 * @param adc_raw_data[in] - ADC raw value
 * @return ADC data after signed conversion
 */
int32_t perform_sign_conversion(uint32_t adc_raw_data)
{
	int32_t adc_data;
	bool bipolar = ad4110_dev_inst->bipolar;

	if (bipolar) {
		adc_data =  adc_raw_data - ADC_MAX_COUNT_BIPOLAR;
	} else {
		adc_data = adc_raw_data;
	}

	return adc_data;
}


/*!
 * @brief Convert the ADC raw value into equivalent RTD resistance
 * @param adc_raw[in] - ADC raw sample
 * @param rtd_ref[in] - RTD reference resistance in ohms
 * @return RTD resistance value
 * @note RTD is biased with constant excitation current. Below formula
 *			is based on ratiometric measurement, where fixed value of RTD RREF
 *			(reference resistor) and gain is taken into account
 */
float convert_adc_raw_into_rtd_resistance(uint32_t adc_raw, float rtd_ref)
{
	int32_t adc_data;
	bool bipolar = ad4110_dev_inst->bipolar;

	adc_data = perform_sign_conversion(adc_raw);

	if (bipolar) {
		return (((float)adc_data * rtd_ref) / (ADC_MAX_COUNT_BIPOLAR *
						       AD4110_DEFAULT_PGA));
	} else {
		return (((float)adc_data * rtd_ref) / (ADC_MAX_COUNT_UNIPOLAR *
						       AD4110_DEFAULT_PGA));
	}
}


/*!
 * @brief Convert the ADC raw value into equivalent voltage
 * @param adc_raw[in]- ADC raw data
 * @param channel_id - Channel ID (number)
 * @return ADC voltage value
 */
float convert_adc_sample_into_voltage(uint32_t adc_raw, uint8_t channel_id)
{
	int32_t adc_data;
	bool bipolar = ad4110_dev_inst->bipolar;
	uint8_t gain = 0;

	adc_data = perform_sign_conversion(adc_raw);

	/* NOTE: PGA can be configured only for channel-0. The gain for the
	 * other channels (LV Channels) is always 1 */
	if (channel_id == HV_CHANNEL) {
		gain = AD4110_DEFAULT_PGA;
	} else {
		gain = AD4110_LV_CHANNEL_PGA;
	}

	if (bipolar) {
		return (adc_data * (AD4110_REF_VOLTAGE / (ADC_MAX_COUNT_BIPOLAR * gain)));
	} else {
		return (adc_data * (AD4110_REF_VOLTAGE / (ADC_MAX_COUNT_UNIPOLAR * gain)));
	}
}