/*!
 ******************************************************************************
 * @file:   adi_sense_log.c
 * @brief:  ADI Sense OS-dependent wrapper layer for log functions
 *-----------------------------------------------------------------------------
 */

/******************************************************************************
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *****************************************************************************/

#include <mbed.h>
#include "inc/adi_sense_log.h"

#define ADI_SENSE_LOG_UART_BAUDRATE (115200)

#ifdef TARGET_STM32F411xE
#define ADI_SENSE_LOG_UART_TX_PIN   (PA_11)
#define ADI_SENSE_LOG_UART_RX_PIN   (PA_12)
#else
#error "unsupported target platform"
#endif

static Serial *gpUartDevice;

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Open the Log interface and allocate resources
 */
ADI_SENSE_RESULT adi_sense_LogOpen(void)
{
    gpUartDevice = new Serial(ADI_SENSE_LOG_UART_TX_PIN,
                              ADI_SENSE_LOG_UART_RX_PIN,
                              ADI_SENSE_LOG_UART_BAUDRATE);
    if (!gpUartDevice)
    {
        ADI_SENSE_LOG_ERROR("Failed to allocate memory for Log UART context");
        return ADI_SENSE_NO_MEM;
    }

    return ADI_SENSE_SUCCESS;
}

/*
 * Close the Log interface and free resources
 */
void adi_sense_LogClose(void)
{
    if (gpUartDevice)
    {
        delete gpUartDevice;
        gpUartDevice = 0;
    }
}

/*
 * Print a log message to the platform log interface
 */
void adi_sense_Log(ADI_SENSE_LOG_LEVEL level, const char* format, ...)
{
    char buffer[256];
    va_list va_args;

    if (!gpUartDevice)
        return;

    va_start(va_args, format);
    vsnprintf(buffer, sizeof(buffer), format, va_args);
    va_end(va_args);

    gpUartDevice->printf("%s\r\n", buffer);
}

#ifdef __cplusplus
}
#endif

