/**  
 @file       ADE120x.h
 @brief      ADE120x library. This file contains all ADE120x library functions. 
 @version    V0.0.1
 @author     ADI
 @date       October 2019
 
 Copyright (c) 2017-2019 Analog Devices, Inc. All Rights Reserved.
 
 This software is proprietary to Analog Devices, Inc. and its licensors.
 By using this software you agree to the terms of the associated
 Analog Devices Software License Agreement. 
*****************************************************************************/

#ifndef _ADE120x_h_
#define _ADE120x_h_

#include "stdio.h"
#include "math.h"
#include "string.h"
#include <stdint.h>

#include "mbed.h"


 /**
 * Structure for configuring threshold levels
*/ 
typedef struct threshold{
    float BIN_HighThresh;   /** Binary channel high threshold */
    float BIN_LowThresh;    /** Binary channel low threshold */
    float WARNA_HighThresh; /** WARNA channel high threshold */
    float WARNA_LowThresh;  /** WARNA channel Low threshold */
    float WARNB_HighThresh; /** WARNB channel high threshold */
    float WARNB_LowThresh;  /** WARNB channel Low threshold */
    float WARNC_HighThresh; /** WARNC channel high threshold */
    float WARNC_LowThresh;  /** WARNC channel Low threshold */
    uint8_t BIN_Mode;       /** Binary Channel comparator mode */
    uint8_t WARNA_Mode;     /** WARNA Channel comparator mode */
    uint8_t WARNB_Mode;     /** WARNB Channel comparator mode */
    uint8_t WARNC_Mode;     /** WARNC Channel comparator mode */
    uint8_t ADCPga;         /** ADC PGA setting */
    float VGain;            /** Voltage divider gain */
}THRESHCfg_Type;

/**
 * Structure for setting programmable load
*/  
typedef struct{
    uint8_t enable;         /** Enable PL */
    uint8_t mode;           /** Configure mode, HIGH_IDLE, LOW_IDLE */
    float HighCurrent;      /** High current in mA */
    float LowCurrent;       /** Low current in mA */
    float HighTime;         /** Duration of current pulse in us */
    float VoltThresh;       /** VOltage threshold to trigger programmable load */
    uint8_t ADCPga;         /** ADC PGA setting */
    float VGain;            /** Voltage gain set by external resister divider */
}PLOADCfg_Type;

/**
 * Structure for configuring energy meter
*/
typedef struct{
    uint8_t enable;             /** Enable energy meter function */
    float FET_Energy;           /** SOA energy of exernal FET used */
    float PulseMagnitude;       /** AMplitude of current pulse in mA */
    float PulseTime;            /** Length of current pulse in ms */
    float WorkingVoltage;       /** Working voltage of system in V*/
    uint32_t AvgADCCode;        /** Average ADC code */
    float SampleRate;           /** Sample rate, 20us for ADE1202, 10us foe ADE1201 */  
    uint8_t Cooldown_TimeStep;  /** Set cooldown timestep */
    uint8_t Cooldown_Decr;      /* Set cooldown decrement */
    uint8_t Cooldown_Sec;       /** Set cooldown period */
    uint8_t Ov_Scale;           /** Set over voltage scale factor */
    uint8_t ADCPga;             /** ADC PGA setting */
    float VGain;                /** Voltage gain set by external resister divider */
}EnergyMtrCfg_Type;

/**
 * Data sreuture for reading back register data
*/
typedef struct{
    uint16_t reg_addr;
    uint32_t reg_data;
}RegisterData_Type;

/******* REGISTER DEFINITION ***********/

#define REG_LOCK            0x000
#define REG_CTRL            0x001
#define REG_BIN_CTRL        0x002
#define REG_BIN_THR         0x003
#define REG_WARNA_THR       0x004
#define REG_WARNB_THR       0x005
#define REG_WARNC_THR       0x006
#define REG_BIN_FILTER      0x007
#define REG_WARNA_FILTER    0x008
#define REG_WARNB_FILTER    0x009
#define REG_WARNC_FILTER    0x00A
#define REG_MASK            0x00B
#define REG_INT_STATUS      0x00C
#define REG_STATUS          0x00D
#define REG_ADC             0x00E
#define REG_ADCDEC          0x00F
#define REG_PL_CTRL         0x010
#define REG_PL_RISE_THR     0x011
#define REG_PL_LOW_CODE     0x012
#define REG_PL_HIGH_CODE    0x013
#define REG_PL_HIGH_TIME    0x014
#define REG_EGY_MTR_CTRL    0x015
#define REG_EGY_MTR_THR     0x016
#define REG_EGY_MTR1        0x017
#define REG_PL_EN           0x200
#define REG_PGA_GAIN        0x201

/**************************************/

/******** BIT DEFINITION **************/
/**     Config_LOCK         **/
#define DEV_UNLOCK          0xADE0
#define DEV_LOCK            0xADE1

/**     CTRL                **/
#define DEV_ADE1201         0x0
#define DEV_ADE1202         0x1000
#define SW_RST              0x10
#define ADDR_RELOAD         0x8
#define ADE1202_IRQ         0x4
#define CRC_EN              0x1

/**     BIN_CTRL            **/
#define Mode_Hysteretic     0
#define Mode_Inbetween      1
#define Mode_Greater        2
#define Mode_LessEqual      3
#define Decrate_Bypass      0
#define Decrate_2           1
#define Decrate_4           2
#define Decrate_8           3

/**     xxx_FILTER          **/
#define Filter_En           (1<<15)
#define Filter_UpDown       (1<<14)

/** Interrupt source selection. These sources are defined as bit mask. 
 *  They are available for register INT_STATUS and STATUS  **/
 #define INTSRC_DOUT1       0x0001      /**<  Bit0, DOUT1 */
 #define INTSRC_WARNA1      0x0002      /**<  Bit1, Warning A from channel 1 */
 #define INTSRC_WARNB1      0x0004      /**<  Bit2, Warning B from channel 1 */
 #define INTSRC_WARNC1      0x0008      /**<  Bit3, Warning C from channel 1 */
 #define INTSRC_DOUT2       0x0010      /**<  Bit4, DOUT2 */
 #define INTSRC_WARNA2      0x0020      /**<  Bit5, Warning A from channel 2 */
 #define INTSRC_WARNB2      0x0040      /**<  Bit6, Warning B from channel 2 */
 #define INTSRC_WARNC2      0x0080      /**<  Bit7, Warning C from channel 2 */
 #define INTSRC_MEMFLT      0x0100      /**<  Bit8, Memory fault. After a memory fault is detected the user could reconfigure the device. */
 #define INTSRC_COMFLT      0x0200      /**<  Bit9, Communication fault */
 #define INTSRC_TSD         0x0400      /**<  Bit10, Thermal shutdown detected */
 #define INTSRC_COOLDOWN1   0x0800      /**<  Bit12, Channel 1 is in Cooldown mode */
 #define INTSRC_COOLDOWN2   0x1000      /**<  Bit13, Channel 2 is in Cooldown mode */
 #define INTSRC_BUSY        0x2000      /**<  Bit13, During busy assertion, internal communication is in progress. Once busy is deasserted, an irq can be triggered which indicates normal operation has resumed */
 #define INTSRC_RSTDONE     0x4000      /**<  Bit14, Indicates that the device has reset and is ready to be programmed or begin default normal operation */
 #define INTSRC_ALL         0x7FFF      /**< All bits. Used to clear all interrupt sources */

/**     ADC                 **/
#define ADC_RAW             0
#define ADC_DECIMATOR       1

/**     PL_CTRL             **/
#define LOW_IDLE            0
#define HIGH_IDLE           1

/**     EGY_MTR_CTRL        **/
#define OV_SCALE_1          0
#define OV_SCALE_4          1
#define OV_SCALE_8          2  
#define OV_SCALE_16         3


typedef enum
{
  COOLDOWN_TS_10us = 0,
  COOLDOWN_TS_20us = 1,
  COOLDOWN_TS_40us = 2,
  COOLDOWN_TS_80us = 3
}COOLDOWN_TIMESTAMP;

#define CH1_Enable              1
#define CH2_Enable              2
#define CH1_CH2_Enable          3
#define CH1_Disable             0
#define CH2_disable             0
#define CH1_CH2_Disable         0

/**     PL_EN                   **/
#define PL_CH2_ENABLE           (1<<15)
#define PL_CH1_ENABLE           (1<<14)
#define PL_CH2_DISABLE          (0<<15)
#define PL_CH1_DISABLE          (0<<14)

/**     PGA_GAIN            **/
#define ADCPGA_1            1
#define ADCPGA_2            3
#define ADCPGA_5            7
#define ADCPGA_10           0xF


/**
 * Method to identify ADE120x
 * [15:14][13:12][11:9] [8:5]
 * [ RES ][MODEL][Addr][RevIf]
 * 
*/ 
#define ADE120x_Model(data)         ((((uint32_t)data)>>12)&0x3)    /**< Return model. 0: ADE1201, 1:ADE1202 */
#define ADE120x_ChipAddr(data)      ((((uint32_t)data)>>9)&0x7)     /**< Return Chip address*/
#define ADE120x_RevId(data)         ((((uint32_t)data)>>5)&0xf)     /**< Return silicon rev ID */

class ADE120x{
public:
    ADE120x(PinName mosi, PinName miso, PinName sclk, PinName cs);
    
    void WriteReg(uint8_t addr, uint32_t reg_addr, uint32_t data);
    uint32_t ReadReg(uint8_t addr, uint32_t reg_addr);
    
    
    uint8_t     Reset(uint8_t addr);
    uint16_t    GetDevID(uint8_t addr);
    void        UnLock(uint8_t addr);
    void        Lock(uint8_t addr);
    uint8_t     DefaultConfig(uint8_t addr);
    
    void        ClearIntStatus(uint8_t addr, uint16_t IntSrcSel);
    uint16_t    GetIntStatus(uint8_t addr);
    void        SetInt(uint8_t addr, uint16_t IntSrcSel);
    
    void        SetBinaryThresh(uint8_t addr, uint16_t thresh);
    uint8_t     CalculateThreshCode(float V_Thresh, uint8_t ADCPga, float V_Gain);
    uint8_t     ThresholdCfg(uint8_t addr, THRESHCfg_Type *pCfg);
    uint8_t     ProgrammableLoadCfg(uint8_t addr, PLOADCfg_Type *pCfg);
    uint8_t     EnergyMtrCfg(uint8_t addr, EnergyMtrCfg_Type *pCfg);
    
    void        SetPgaGain(uint8_t addr, uint16_t gain);
    uint8_t     ReadADC(uint8_t addr, int8_t src);
    float       ADCCode2Volt(uint32_t ADCCode, uint8_t ADCPga, float VOLTAGE_Gain);
    void        GetRegisterData(uint8_t addr, RegisterData_Type *pBuff);
private:
    SPI        spi_;
    DigitalOut nCS_;

};

#endif
