/* 
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2015 Semtech

Description: MBED LoRaWAN example application

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainer: Miguel Luis and Gregory Cristian
*/

#include <cstdio>
#include <string>
#include <cassert>

#include "mbed.h"
//#include "Node.h"
#include "cantcoap.h"
#include "x_nucleo_iks01a1.h"
#include "rtos.h"

#include "lmic.h"
#include "debug.h"

/* Instantiate the expansion board */
static X_NUCLEO_IKS01A1 *mems_expansion_board = X_NUCLEO_IKS01A1::Instance(D14, D15);

/* Retrieve the composing elements of the expansion board */
static GyroSensor *gyroscope = mems_expansion_board->GetGyroscope();
static MotionSensor *accelerometer = mems_expansion_board->GetAccelerometer();
static MagneticSensor *magnetometer = mems_expansion_board->magnetometer;
static HumiditySensor *humidity_sensor = mems_expansion_board->ht_sensor;
static PressureSensor *pressure_sensor = mems_expansion_board->pt_sensor;
static TempSensor *temp_sensor1 = mems_expansion_board->ht_sensor;
static TempSensor *temp_sensor2 = mems_expansion_board->pt_sensor;

uint8_t id;
float value1, value2;
char buffer1[32], buffer2[32], buffer3[32];
int32_t axes[3];

static bool temp_sent = false;
static bool hum_sent = false;
static bool press_sent = false;
static bool gyro_sent = false;
static bool motion_sent = false;
static bool magn_sent = false;

const std::string REGISTRATION_SEGMENT ="/rd";
const std::string ENDPOINT_SEGMENT     = "?ep=";
const std::string LIFETIME             ="&lt=";
const std::string BINDING              ="&b=";
 
const std::string REGISTRATION_OPEN  =    "<";
const std::string REGISTRATION_CLOSE =    ">";
const std::string REGISTRATION_SEPARATOR ="/";
 
int _node_Id=0;
 
const std::string endPoint_Name = "loraDevice"; 
const int lifeTime = 300;
const std::string binding = "U";
 
unsigned char * _payload;
long _payload_size;
 
/*!
 * When set to 1 the application uses the Over-the-Air activation procedure
 * When set to 0 the application uses the Personalization activation procedure
 */
#define OVER_THE_AIR_ACTIVATION                     1

#if( OVER_THE_AIR_ACTIVATION == 0 )

/*!
 * Defines the network ID when using personalization activation procedure
 */
#define LORAWAN_NET_ID                              ( uint32_t )0x00000000

/*!
 * Defines the device address when using personalization activation procedure
 */
//To be changed when switching from one to another
#define LORAWAN_DEV_ADDR                            ( uint32_t )0x12342222

#endif

/*!
 * Defines the application data transmission duty cycle
 */
//#define APP_TX_DUTYCYCLE                            5000 // 5 [s] value in ms
//PANY
#define APP_TX_DUTYCYCLE                            1000 // 1 [s] value in ms
//PANY
#define APP_TX_DUTYCYCLE_RND                        1000 // 1 [s] value in ms

/*!
 * LoRaWAN Adaptative Data Rate
 */
#define LORAWAN_ADR_ON                              1

/*!
 * LoRaWAN confirmed messages
 */
#define LORAWAN_CONFIRMED_MSG_ON                    0

/*!
 * LoRaWAN application port
 */
#define LORAWAN_APP_PORT                            15

/*!
 * User application data buffer size
 */
#if ( LORAWAN_CONFIRMED_MSG_ON == 1 )
#define LORAWAN_APP_DATA_SIZE                       6

#else
#define LORAWAN_APP_DATA_SIZE                       1

#endif

#define UINT16_MAX (65535U)
#define UINT64_MAX (18446744073709551615ULL)
 
//Node lwm2mNode("LR-test0");;
unsigned int LoRaWAN_data_size = 0; 

/* Helper function for printing floats & doubles */
static char *printDouble(char* str, double v, int decimalDigits=2)
{
  int i = 1;
  int intPart, fractPart;
  int len;
  char *ptr;

  /* prepare decimal digits multiplicator */
  for (;decimalDigits!=0; i*=10, decimalDigits--);

  /* calculate integer & fractinal parts */
  intPart = (int)v;
  fractPart = (int)((v-(double)(int)v)*i);

  /* fill in integer part */
  sprintf(str, "%i.", intPart);

  /* prepare fill in of fractional part */
  len = strlen(str);
  ptr = &str[len];

  /* fill in leading fractional zeros */
  for (i/=10;i>1; i/=10, ptr++) {
    if(fractPart >= i) break;
    *ptr = '0';
  }

  /* fill in (rest of) fractional part */
  sprintf(ptr, "%i", fractPart);

  return str;
}

/* Helper function for printing integers */
static char *printInt(char* str, int v)
{
  /* fill in integer part */
  sprintf(str, "%i", v);

  return str;
}

/* Thread for calling libNsdl exec function (cleanup, resendings etc..) */
/* Node updates accelerometer every 60 seconds. Notification sending is done here. */
static void exec_call_thread(void const *args)
{
  uint8_t id;
  float value1, value2;
  char buffer1[32], buffer2[32];
  int32_t axes[3];
  
  debug_str("--- Starting new run ---\r\n");

  humidity_sensor->ReadID(&id);
  debug_str("HTS221  humidity & temperature    = ");
  debug_uint(id);
  debug_str("\r\n");
  pressure_sensor->ReadID(&id);
  debug_str("LPS25H  pressure & temperature    = ");
  debug_uint(id);
  debug_str("\r\n");
  magnetometer->ReadID(&id);
  debug_str("LIS3MDL magnetometer              = ");
  debug_uint(id);
  debug_str("\r\n");
  gyroscope->ReadID(&id);
  debug_str("LSM6DS0 accelerometer & gyroscope = ");
  debug_uint(id);
  debug_str("\r\n");
  
  wait(3);
 
  while(1) {
    debug_str("\r\n");

    temp_sensor1->GetTemperature(&value1);
    humidity_sensor->GetHumidity(&value2);
    debug_str("HTS221: [temp] ");
    debug_str(printDouble(buffer1, value1));
    debug_str("°C,   [hum] ");
    debug_str(printDouble(buffer2, value2));
    debug_str("%\r\n");
    //pc.printf("HTS221: [temp] %7s°C,   [hum] %s%%\r\n", printDouble(buffer1, value1), printDouble(buffer2, value2));
    
    temp_sensor2->GetFahrenheit(&value1);
    pressure_sensor->GetPressure(&value2);
    debug_str("LPS25H: [temp] ");
    debug_str(printDouble(buffer1, value1));
    debug_str("°F, [press] ");
    debug_str(printDouble(buffer2, value2));
    debug_str("mbar\r\n");
    //pc.printf("LPS25H: [temp] %7s°F, [press] %smbar\r\n", printDouble(buffer1, value1), printDouble(buffer2, value2));

    debug_str("---\r\n");

    magnetometer->Get_M_Axes(axes);
    debug_str("LIS3MDL [mag/mgauss]:  ");
    debug_uint(axes[0]);
    debug_str(", ");
    debug_uint(axes[1]);
    debug_str(", ");
    debug_uint(axes[2]);
    debug_str("\r\n");
    //pc.printf("LIS3MDL [mag/mgauss]:  %6ld, %6ld, %6ld\r\n", axes[0], axes[1], axes[2]);

    accelerometer->Get_X_Axes(axes);
    debug_str("LSM6DS0 [acc/mg]:      ");
    debug_uint(axes[0]);
    debug_str(", ");
    debug_uint(axes[1]);
    debug_str(", ");
    debug_uint(axes[2]);
    debug_str("\r\n");
    //pc.printf("LSM6DS0 [acc/mg]:      %6ld, %6ld, %6ld\r\n", axes[0], axes[1], axes[2]);

    gyroscope->Get_G_Axes(axes);
    debug_str("LSM6DS0 [gyro/mdps]:   ");
    debug_uint(axes[0]);
    debug_str(", ");
    debug_uint(axes[1]);
    debug_str(", ");
    debug_uint(axes[2]);
    debug_str("\r\n");
    //pc.printf("LSM6DS0 [gyro/mdps]:   %6ld, %6ld, %6ld\r\n", axes[0], axes[1], axes[2]);

    wait(1.5);
  }
}

std::string to_string( int x ) {
  int length = snprintf( NULL, 0, "%d", x );
  //assert( length >= 0 );
  char* buf = new char[length + 1];
  snprintf( buf, length + 1, "%d", x );
  std::string str( buf );
  delete[] buf;
  return str;
}

unsigned char * get_Registration_Payload(long *payload_size){
 
    string registration_Payload ="";
 
    string s="";
 
    s.append(REGISTRATION_OPEN);
    s.append(REGISTRATION_SEPARATOR);
    s.append("3/0/0");
    s.append(REGISTRATION_CLOSE);
    s.append(",");
    s.append(REGISTRATION_OPEN);
    s.append(REGISTRATION_SEPARATOR);
    s.append("3/0/1");
    s.append(REGISTRATION_CLOSE);
    s.append(",");
    s.append(REGISTRATION_OPEN);
    s.append(REGISTRATION_SEPARATOR);
    s.append("3/0/2");
    s.append(REGISTRATION_CLOSE);
 
    registration_Payload.append(s);
    
    unsigned char *c = new unsigned char[registration_Payload.size()+1];
    copy(registration_Payload.begin(),registration_Payload.end(),c);
    c[registration_Payload.size()]='\0';
    *payload_size=registration_Payload.size();
    
    return c;
 
}
uint8_t * get_Token(int * size){
    srand(time(0)+_node_Id);
    long test=0;
    bool exist=false;
    
    do{
        test=(rand() % UINT64_MAX);
        
    }while (exist==true);
    uint8_t ones = 0xFF;
    *size=1;
    for (int i=0; i<8; ++i) {
        if ( (test>>8*i & ones) =='\0' || i==8) {
            *size=i;
            break;
        }
    }
    uint8_t * token =new uint8_t[*size];
    for (int i=0; i<*size; ++i){
        token[*size-1-i]=test>>8*i & ones;
    }
    return token;
}
 
uint16_t get_Message_ID(){
    srand(time(0)+_node_Id);
    int test=0;
    bool exist=false;
    do{
        
        exist=false;
        test=(rand() % UINT16_MAX);
        
    }while (exist==true);
    
    
    return (uint16_t) test;
    
}
 
char * get_Registration_Query(){
    
    string buffer;
    buffer.append(REGISTRATION_SEGMENT);
    buffer.append(ENDPOINT_SEGMENT);
    buffer.append(endPoint_Name);
    buffer.append(LIFETIME);
    buffer.append(to_string(lifeTime));
    buffer.append(BINDING);
    buffer.append(binding);
    
    char *c = new char[buffer.size()+1];
    copy(buffer.begin(),buffer.end(),c);
    c[buffer.size()]='\0';
    return c;
    
 
}
 
//////////////////////////////////////////////////
// CONFIGURATION (FOR APPLICATION CALLBACKS BELOW)
//////////////////////////////////////////////////

// application router ID (LSBF)
//To be changed when switching from one to another
static const uint8_t AppEui[8] =
{
//    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    0x01, 0x02, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00
};

// unique device ID (LSBF)
//To be changed when switching from one to another
static const u1_t DevEui[8] =
{
      0x31, 0x74, 0x73, 0x65, 0x74, 0x2D, 0x52, 0x4C // 4c522d7465737433 = "LR-test3"
};

// device-specific AES key (derived from device EUI)
static const uint8_t DevKey[16] = 
{
    0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
    0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3E
};

#if( OVER_THE_AIR_ACTIVATION == 0 )
// network session key
//To be changed when switching from one to another
static uint8_t NwkSKey[] = 
{ 
    0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
    0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3E
};

// application session key
//To be changed when switching from one to another
static uint8_t ArtSKey[] = 
{ 
    0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
    0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3E
};

#endif

// LEDs and Frame jobs
osjob_t rxLedJob;
osjob_t txLedJob;
osjob_t sendFrameJob;

// LED state
static bool AppLedStateOn = false;

//////////////////////////////////////////////////
// Utility functions
//////////////////////////////////////////////////
/*!
 * \brief Computes a random number between min and max
 *
 * \param [IN] min range minimum value
 * \param [IN] max range maximum value
 * \retval random random value in range min..max
 */
int32_t randr( int32_t min, int32_t max )
{
    return ( int32_t )rand( ) % ( max - min + 1 ) + min;
}

//////////////////////////////////////////////////
// APPLICATION CALLBACKS
//////////////////////////////////////////////////

// provide application router ID (8 bytes, LSBF)
void os_getArtEui( uint8_t *buf )
{
    memcpy( buf, AppEui, 8 );
}

// provide device ID (8 bytes, LSBF)
void os_getDevEui( uint8_t *buf )
{
    memcpy( buf, DevEui, 8 );
}

// provide device key (16 bytes)
void os_getDevKey( uint8_t *buf )
{
    memcpy( buf, DevKey, 16 );
}

//////////////////////////////////////////////////
// MAIN - INITIALIZATION AND STARTUP
//////////////////////////////////////////////////

static void onRxLed( osjob_t* j )
{
//    debug_val("LED2 = ", 0 );
}

static void onTxLed( osjob_t* j )
{
//    debug_val("LED1 = ", 0 );
}

static void prepareTxCoapFrame( void )
{
    // Create Registration PDU :
    
   CoapPDU *pdu = new CoapPDU();
    
    pdu->setCode(CoapPDU::COAP_POST);
    pdu->setType(CoapPDU::COAP_CONFIRMABLE);
    int size;
    uint8_t * token = get_Token(&size);
    pdu->setToken(token,size);
    pdu->setMessageID(get_Message_ID());
    pdu->setURI(get_Registration_Query());
    
    _payload=get_Registration_Payload(&_payload_size);
    pdu->setPayload(_payload, (int) _payload_size);
    int PDUlength = pdu->getPDULength();
    
//    strncpy((char*) LMIC.frame, (const char*)pdu->getPDUPointer(), PDUlength);
    memcpy(LMIC.frame, pdu->getPDUPointer(), PDUlength * sizeof(uint8_t));

#if ( LORAWAN_CONFIRMED_MSG_ON == 1 )
    LMIC.frame[PDUlength] = LMIC.seqnoDn >> 8;
    LMIC.frame[PDUlength+1] = LMIC.seqnoDn;
    LMIC.frame[PDUlength+2] = LMIC.rssi >> 8;
    LMIC.frame[PDUlength+3] = LMIC.rssi;
    LMIC.frame[PDUlength+4] = LMIC.snr;
#endif
    debug_str("Frame to be sent: ");
    debug_buf(LMIC.frame, PDUlength + 5);
    
    LoRaWAN_data_size = PDUlength + 5;
}

static void prepareTxLoraFrame( void )
{
    const char *frame = "LoRa";
//    const char *frame = "Test";
    
    strncpy((char*) LMIC.frame, frame, strlen(frame));

#if ( LORAWAN_CONFIRMED_MSG_ON == 1 )
    LMIC.frame[strlen(frame)] = LMIC.seqnoDn >> 8;
    LMIC.frame[strlen(frame)+1] = LMIC.seqnoDn;
    LMIC.frame[strlen(frame)+2] = LMIC.rssi >> 8;
    LMIC.frame[strlen(frame)+3] = LMIC.rssi;
    LMIC.frame[strlen(frame)+4] = LMIC.snr;
#endif
    debug_str("Frame to be sent: ");
//    debug_buf(LMIC.frame, strlen(frame) + 5);
    debug_buf(LMIC.frame, strlen(frame));
    
//    LoRaWAN_data_size = strlen(frame) + 5;
    LoRaWAN_data_size = strlen(frame);
}

static void prepareTxSensorsFrame( void )
{
    std::string frame = "";
    std::string tmp;
//    debug_str("\r\n");

    temp_sensor1->GetTemperature(&value1);
    humidity_sensor->GetHumidity(&value2);
    printDouble(buffer1, value1);
//    debug_str("HTS221: [temp] ");
//    debug_str(printDouble(buffer1, value1));
    tmp = "0,";
    tmp += buffer1;
    tmp += ",";
    
    if ((!temp_sent) && (frame.length() + tmp.length() < 50))
    {
        temp_sent = true;
        frame += tmp; 
    }
    printDouble(buffer2, value2);
//    debug_str("Celsius,   [hum] ");
//    debug_str(printDouble(buffer2, value2));
    tmp = "1,";
    tmp += buffer2;
    tmp += ",";

    if ((!hum_sent) && (frame.length() + tmp.length() < 50))
    {
        hum_sent = true;
        frame += tmp; 
    }
    
//    debug_str("%\r\n");
    //pc.printf("HTS221: [temp] %7s°C,   [hum] %s%%\r\n", printDouble(buffer1, value1), printDouble(buffer2, value2));
    
    temp_sensor2->GetFahrenheit(&value1);
    pressure_sensor->GetPressure(&value2);
    printDouble(buffer1, value1);
    printDouble(buffer2, value2);
/*    debug_str("LPS25H: [temp] ");
    debug_str(printDouble(buffer1, value1));
    debug_str("Farenheit, [press] ");
    debug_str(printDouble(buffer2, value2));*/
    tmp = "2,";
    tmp += buffer2;
    tmp += ",";

    if ((!press_sent) && (frame.length() + tmp.length() < 50))
    {
        press_sent = true;
        frame += tmp; 
    }
    
//    debug_str("mbar\r\n");
    //pc.printf("LPS25H: [temp] %7s°F, [press] %smbar\r\n", printDouble(buffer1, value1), printDouble(buffer2, value2));

//    debug_str("---\r\n");

    magnetometer->Get_M_Axes(axes);
    printInt(buffer1, axes[0]);
    printInt(buffer2, axes[1]);
    printInt(buffer3, axes[2]);
/*    debug_str("LIS3MDL [mag/mgauss]:  ");
    debug_str(printInt(buffer1, axes[0]));
    debug_str(", ");
    debug_str(printInt(buffer2, axes[1]));
    debug_str(", ");
    debug_str(printInt(buffer3, axes[2]));
    debug_str("\r\n");*/
    tmp = "3,";
    tmp += buffer1;
    tmp += ";";
    tmp += buffer2;
    tmp += ";";
    tmp += buffer3;
    tmp += ",";

    if ((!magn_sent) && (frame.length() + tmp.length() < 50))
    {
        magn_sent = true;
        frame += tmp; 
    }
    
    //pc.printf("LIS3MDL [mag/mgauss]:  %6ld, %6ld, %6ld\r\n", axes[0], axes[1], axes[2]);

    accelerometer->Get_X_Axes(axes);
    printInt(buffer1, axes[0]);
    printInt(buffer2, axes[1]);
    printInt(buffer3, axes[2]);
/*   debug_str("LSM6DS0 [acc/mg]:      ");

    debug_str(printInt(buffer1, axes[0]));
    debug_str(", ");
    debug_str(printInt(buffer2, axes[1]));
    debug_str(", ");
    debug_str(printInt(buffer3, axes[2]));*/
    tmp = "4,";
    tmp += buffer1;
    tmp += ";";
    tmp += buffer2;
    tmp += ";";
    tmp += buffer3;
    tmp += ",";

    if ((!motion_sent) && (frame.length() + tmp.length() < 50))
    {
        motion_sent = true;
        frame += tmp; 
    }
    
    //debug_str("\r\n");
    //pc.printf("LSM6DS0 [acc/mg]:      %6ld, %6ld, %6ld\r\n", axes[0], axes[1], axes[2]);

    gyroscope->Get_G_Axes(axes);
    printInt(buffer1, axes[0]);
    printInt(buffer2, axes[1]);
    printInt(buffer3, axes[2]);
/*    debug_str("LSM6DS0 [gyro/mdps]:   ");
    debug_str(printInt(buffer1, axes[0]));
    debug_str(", ");
    debug_str(printInt(buffer2, axes[1]));
    debug_str(", ");
    debug_str(printInt(buffer3, axes[2]));*/
    tmp = "5,";
    tmp += buffer1;
    tmp += ";";
    tmp += buffer2;
    tmp += ";";
    tmp += buffer3;

    if ((!gyro_sent) && (frame.length() + tmp.length() < 50))
    {
        gyro_sent = false;
        temp_sent = false;
        press_sent = false;
        motion_sent = false;
        magn_sent = false;
        hum_sent = false;
        
        frame += tmp; 
    }
    
    //debug_str("\r\n Frame: ");
    //debug_str(frame.c_str());
    //debug_str("\r\n");
    //pc.printf("LSM6DS0 [gyro/mdps]:   %6ld, %6ld, %6ld\r\n", axes[0], axes[1], axes[2]);

    strncpy((char*) LMIC.frame, frame.c_str(), strlen(frame.c_str()));

#if ( LORAWAN_CONFIRMED_MSG_ON == 1 )
    LMIC.frame[strlen(frame.c_str())] = LMIC.seqnoDn >> 8;
    LMIC.frame[strlen(frame.c_str())+1] = LMIC.seqnoDn;
    LMIC.frame[strlen(frame.c_str())+2] = LMIC.rssi >> 8;
    LMIC.frame[strlen(frame.c_str())+3] = LMIC.rssi;
    LMIC.frame[strlen(frame.c_str())+4] = LMIC.snr;
#endif
    //debug_str("Frame to be sent: ");
//    debug_buf(LMIC.frame, strlen(frame) + 5);
    //debug_buf(LMIC.frame, strlen(frame.c_str()));
    
//    LoRaWAN_data_size = strlen(frame) + 5;
    LoRaWAN_data_size = strlen(frame.c_str());
}

static void prepareTxFrame( void )
{
    LMIC.frame[0] = AppLedStateOn;
#if ( LORAWAN_CONFIRMED_MSG_ON == 1 )
    LMIC.frame[1] = LMIC.seqnoDn >> 8;
    LMIC.frame[2] = LMIC.seqnoDn;
    LMIC.frame[3] = LMIC.rssi >> 8;
    LMIC.frame[4] = LMIC.rssi;
    LMIC.frame[5] = LMIC.snr;
#endif    
    //debug_str("Frame to be sent: ");
    //debug_buf(LMIC.frame, LORAWAN_APP_DATA_SIZE);
    
    LoRaWAN_data_size = LORAWAN_APP_DATA_SIZE;
}

void processRxFrame( void )
{

    char* frameToDisplay = (char*) (LMIC.frame + LMIC.dataBeg);
    frameToDisplay[LMIC.dataLen] = '\0';

    switch( LMIC.frame[LMIC.dataBeg - 1] ) // Check Rx port number
    {
        case 0:
//            debug_str("Port 0!!!\r\n");
//            debug_val("Data Len: ", LMIC.dataLen);
            
        case 1: // The application LED can be controlled on port 1 or 2
            debug_str("Data received on port 1: ");
            debug_str("Data in hexa: ");
            debug_buf( LMIC.frame + LMIC.dataBeg, LMIC.dataLen );
            debug_str("Data in string: ");
            
            debug_str( frameToDisplay );
            debug_str("\r\n");
            
            break;
        case 2:
            debug_str("Data received on port 2: ");
            debug_str("Data in hexa: ");
            debug_buf( LMIC.frame + LMIC.dataBeg, LMIC.dataLen );
            debug_str("Data in string: ");
            
            debug_str( frameToDisplay );
            debug_str("\r\n");
        
            if( LMIC.dataLen == 1 )
            {
                debug_str("Data received on port 2: ");
                debug_hex(LMIC.frame[LMIC.dataBeg]);
                debug_str("\r\n");
                AppLedStateOn = LMIC.frame[LMIC.dataBeg] & 0x01;
                //debug_val( "LED3 = ", AppLedStateOn );
            }
            break;
        default:
            break;
    }
}

static void onSendFrame( osjob_t* j )
{
    //prepareTxFrame( );
    //prepareTxCoapFrame();
    prepareTxSensorsFrame();
    //prepareTxLoraFrame();
#if( OVER_THE_AIR_ACTIVATION == 1 )
    LMIC_setTxData2( LORAWAN_APP_PORT, LMIC.frame, LoRaWAN_data_size, LORAWAN_CONFIRMED_MSG_ON, 0 );
#endif  
#if( OVER_THE_AIR_ACTIVATION == 0 )
    LMIC_setTxData2( LORAWAN_APP_PORT, LMIC.frame, LoRaWAN_data_size, LORAWAN_CONFIRMED_MSG_ON, 0 );
#endif  

    // Blink Tx LED
    //debug_val( "LED1 = ", 1 );
    os_setTimedCallback( &txLedJob, os_getTime( ) + ms2osticks( 25 ), onTxLed );
    /*    os_setTimedCallback( &sendFrameJob,
                             os_getTime( ) + ms2osticks( APP_TX_DUTYCYCLE + randr( -APP_TX_DUTYCYCLE_RND, APP_TX_DUTYCYCLE_RND ) ),
                             onSendFrame );*/
}

// Initialization job
static void onInit( osjob_t* j )
{
    debug_str("--- Starting new run ---\r\n");

    humidity_sensor->ReadID(&id);
    debug_str("HTS221  humidity & temperature    = ");
    debug_uint(id);
    debug_str("\r\n");
    pressure_sensor->ReadID(&id);
    debug_str("LPS25H  pressure & temperature    = ");
    debug_uint(id);
    debug_str("\r\n");
    magnetometer->ReadID(&id);
    debug_str("LIS3MDL magnetometer              = ");
    debug_uint(id);
    debug_str("\r\n");
    gyroscope->ReadID(&id);
    debug_str("LSM6DS0 accelerometer & gyroscope = ");
    debug_uint(id);
    debug_str("\r\n");

    // reset MAC state
    LMIC_reset( );
    LMIC_setAdrMode( LORAWAN_ADR_ON );
    LMIC_setDrTxpow( DR_SF12, 14 );

    // start joining
#if( OVER_THE_AIR_ACTIVATION != 0 )
    LMIC_startJoining( );
#else
    LMIC_setSession( LORAWAN_NET_ID, LORAWAN_DEV_ADDR, NwkSKey, ArtSKey );
    onSendFrame( NULL );
#endif
    // init done - onEvent( ) callback will be invoked...
}

int main( void )
{
    debug_init();
    osjob_t initjob;
    
    // initialize runtime env
    os_init( );
    // setup initial job
    os_setCallback( &initjob, onInit );
    // execute scheduled jobs and events
    
    //static Thread exec_thread(exec_call_thread);
    
    os_runloop( );
    // (not reached)
}

//////////////////////////////////////////////////
// LMIC EVENT CALLBACK
//////////////////////////////////////////////////
void onEvent( ev_t ev )
{
    bool txOn = false;
    debug_event( ev );

    switch( ev ) 
    {
    // network joined, session established
    case EV_JOINED:
        debug_val( "Net ID = ", LMIC.netid );
        txOn = true;
        break;
    // scheduled data sent (optionally data received)
    case EV_TXCOMPLETE:
        debug_val( "Datarate = ", LMIC.datarate );
        // Check if we have a downlink on either Rx1 or Rx2 windows
        if( ( LMIC.txrxFlags & ( TXRX_DNW1 | TXRX_DNW2 ) ) != 0 )
        {
//            debug_val( "LED2 = ", 1 );
            os_setTimedCallback( &rxLedJob, os_getTime( ) + ms2osticks( 25 ), onRxLed );

            if( LMIC.dataLen != 0 )
            { // data received in rx slot after tx
                //debug_buf( LMIC.frame + LMIC.dataBeg, LMIC.dataLen );
                processRxFrame( );
            }
        }
        txOn = true;
        break;
    default:
        break;
    }
    if( txOn == true )
    {
        //Sends frame every APP_TX_DUTYCYCLE +/- APP_TX_DUTYCYCLE_RND random time (if not duty cycle limited)
        os_setTimedCallback( &sendFrameJob,
                             os_getTime( ) + ms2osticks( APP_TX_DUTYCYCLE + randr( -APP_TX_DUTYCYCLE_RND, APP_TX_DUTYCYCLE_RND ) ),
                             onSendFrame );
        //os_setCallback( &sendFrameJob, onSendFrame );
        ////Sends frame as soon as possible (duty cylce limitations)
        //onSendFrame( NULL );
    }
}


