/*
 * VFD Modular Clock - mbed
 * (C) 2011-14 Akafugu Corporation
 *
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 *
 */

#include "global.h"
#include "button.h"

DigitalIn button1(PinMap::button1);
DigitalIn button2(PinMap::button2);
DigitalIn button3(PinMap::button3);

uint8_t button_count = 3;

uint8_t saved_keystatus = 0;
uint8_t keydown_keys = 0;
uint8_t keyup_keys = 0;
uint8_t keyrepeat_keys = 0;

uint16_t keyboard_counter[3] = {0, 0, 0};
uint8_t button_bit[3] = { 1, 2, 4};

#define REPEAT_SPEED    35
volatile uint8_t repeat_speed = REPEAT_SPEED;

void initialize_buttons()
{
    // enable pullups
    button1.mode(PullUp);
    button2.mode(PullUp);
    button3.mode(PullUp);
}

uint8_t get_keystatus() {
    return (!button1) | (!button2 << 1) | (!button3 << 2);
}

void button_tick()
{
    uint8_t keystatus = (!button1) | (!button2 << 1) | (!button3 << 2);

    keydown_keys |= (uint8_t)(keystatus & ~(saved_keystatus));
    keyup_keys   |= (uint8_t)(~(keystatus) & saved_keystatus);
    saved_keystatus = keystatus;

    for(uint8_t i = 0; i < button_count; i++) {
        if(~(keydown_keys)&button_bit[i])
            ; // Do nothing, no keyrepeat is needed
        else if(keyup_keys&button_bit[i])
            keyboard_counter[i] = 0;
        else {
            if(keyboard_counter[i] >= repeat_speed) {
                keyrepeat_keys |= button_bit[i];
                keyboard_counter[i] = 0;
            }
            
            keyboard_counter[i]++;
        }
    }
}

void get_button_state(struct BUTTON_STATE* buttons)
{
    buttons->b1_keydown = keydown_keys&(button_bit[0]);
    buttons->b1_keyup   = keyup_keys&(button_bit[0]);
    buttons->b1_repeat  = keyrepeat_keys&(button_bit[0]);
    
    if (keyrepeat_keys&(button_bit[0]))
      keyrepeat_keys &= ~(button_bit[0]);

    // Reset if we got keyup
    if(keyup_keys&(button_bit[0])) {
        keydown_keys   &= ~(button_bit[0]);
        keyup_keys     &= ~(button_bit[0]);
        keyrepeat_keys &= ~(button_bit[0]);
        keyboard_counter[0] = 0;
    }

    buttons->b2_keydown = keydown_keys&(button_bit[1]);
    buttons->b2_keyup = keyup_keys&(button_bit[1]);
    buttons->b2_repeat = keyrepeat_keys&(button_bit[1]);

    if (keyrepeat_keys&(button_bit[1]))
      keyrepeat_keys &= ~(button_bit[1]);
    
    // Reset if we got keyup
    if(keyup_keys&(button_bit[1])) {
        keydown_keys   &= ~(button_bit[1]);
        keyup_keys     &= ~(button_bit[1]);
        keyrepeat_keys &= ~(button_bit[1]);
        keyboard_counter[1] = 0;
    }

    buttons->b3_keydown = keydown_keys&(button_bit[2]);
    buttons->b3_keyup   = keyup_keys&(button_bit[2]);
    buttons->b3_repeat  = keyrepeat_keys&(button_bit[2]);
    
    if (keyrepeat_keys&(button_bit[2]))
      keyrepeat_keys &= ~(button_bit[2]);

    // Reset if we got keyup
    if(keyup_keys&(button_bit[2])) {
        keydown_keys   &= ~(button_bit[2]);
        keyup_keys     &= ~(button_bit[2]);
        keyrepeat_keys &= ~(button_bit[2]);
        keyboard_counter[2] = 0;
    }

    buttons->both_held = (keydown_keys&(button_bit[0])) && (keydown_keys&(button_bit[1]));
    buttons->none_held = ~(saved_keystatus)&(button_bit[0]) && ~(saved_keystatus)&((button_bit[1]));
}

