/*
 * VFD Modular Clock - mbed
 * (C) 2011-14 Akafugu Corporation
 *
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 *
 */

#ifndef __VFDDISPLAY_H_
#define __VFDDISPLAY_H_

#include "mbed.h"

#define MESSAGE_LIMIT 96

class VFDDisplay : public Stream {
public:
    enum TimeMode {
        Long = 0,
        Short,
        Extra 
    };
    
    // Controls how display blinks when m_blink is true
    enum BlinkMode {
        Full = 0, // Blink whole display
        Hours,    // Blink hours
        Minutes,  // Blink minutes
        Seconds   // Blink seconds
    };
    
protected:
    TimeMode m_mode;
    BlinkMode m_blink_mode;
    
public:
    VFDDisplay(PinName data, PinName clock, PinName latch, PinName blank, uint8_t digits);

    virtual void cls();
    virtual void setPosition(uint8_t pos);
    
    virtual int putc(int c) { return _putc(c); }
    int printf(const char* format, ...);

    void scroll(int8_t spaces = 1);
    void resetScroll();
    bool scrollFinished();
    bool scrollAtOrigin() { return m_scroll_offset == 0; }

    void setAlarmIndicator(bool on) { m_alarm_indicator = on; }
    void setGPSIndicator(bool on)   { m_gps_indicator   = on; }
    void setColon(bool on)          { m_colon = on; }
    
    void setDots(uint16_t dots) { m_dots = dots; }
    void setDot(uint8_t pos, bool on);

    void setTimeMode(TimeMode mode) { m_mode = mode; }
    void toggleTimeMode();
    
    //virtual void printTime(time_t t, uint8_t hundredths) = 0;
    virtual void printTime(struct tm* tm, uint8_t hundredths) = 0;
    virtual void printTimeLong(struct tm* tm, uint8_t hundredths) = 0;
    virtual void printTimeSet(struct tm* tm, bool showSeconds = true) = 0;
    virtual void printDate(struct tm* tm) = 0;
    
    void setBrightness(uint8_t brite);
    uint8_t getBrightness() { return m_brightness; }
    uint8_t incBrightness();

    uint8_t digits() { return m_digits; }
    
    void setBlinkMode(BlinkMode blink_mode) { m_blink_mode = blink_mode; }
    void blink(bool b) { m_blink = b; m_display_on = true; }
    void blank(bool b) { m_blank = b; }    
    
protected:
    DigitalOut m_data;
    DigitalOut m_clock;
    DigitalOut m_latch;
    //PwmOut m_blank;
    DigitalOut m_blank;

    uint8_t m_digits;
    uint8_t m_multiplex_limit;
    uint8_t m_multiplex_counter;
    bool m_reverse_display;
    uint8_t m_position;
    int8_t m_scroll_offset;
    uint8_t m_message_length;
    bool m_has_dots;

    char m_buffer[MESSAGE_LIMIT];
    char m_dot_buffer[MESSAGE_LIMIT];
    char m_char_buffer[MESSAGE_LIMIT];
        
    void writeHV5812(uint8_t data);
    virtual uint16_t calculateSegments(char c, uint8_t digit) = 0;
    virtual void handleBlink(char d) = 0;
    
    // extra indicators (not present on all displays
    bool m_alarm_indicator;
    bool m_gps_indicator;
    bool m_colon;
    uint16_t m_dots;
    
    // display blinking
    volatile bool m_blink;
    volatile bool m_display_on;
    
    // brightness
    uint8_t m_brightness;

public:
    virtual void writeDisplay(uint8_t digit, uint16_t segments) = 0;
    void multiplexTick();
            
protected:
    virtual int _putc(int c);
    virtual int _getc();
};

#endif // __VFDDISPLAY_H_
