/*
 * VFD Modular Clock - mbed
 * (C) 2011-14 Akafugu Corporation
 *
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 *
 */

#include "global.h"

#include "prefs.h"
#include "DipCortex-EEprom.h"

#include "VFDDisplay.h"
#include "IV18Display.h"
#include "flw.h"

const uint8_t sig0 = 0x42;
const uint8_t sig1 = 0x66;

PrefsData prefs_data;

extern IV18Display display;
extern DigitalOut led;
extern FourLetterWord flw;

static int s_pref_year;
static int s_pref_month;
static int s_pref_day;

uint8_t crc8(const void *vptr, int len)
{
    const uint8_t *data = static_cast<const uint8_t *>(vptr);
    unsigned crc = 0;
    int i, j;
    
    for (j = len; j; j--, data++) {
        crc ^= (*data << 8);
        for(i = 8; i; i--) {
            if (crc & 0x8000)
                crc ^= (0x1070 << 3);
            crc <<= 1;
        }
    }
    
    return (uint8_t)(crc >> 8);
}

void init_prefs()
{
    IAP_Init();
    
    IAP_Eeprom_Read(0, prefs_data.buf, PREFS_SIZE);
    
    bool initialized = false;
    
    // check if eeprom has been initialized
    if (prefs_data.buf[0] == sig0 && prefs_data.buf[1] == sig1) {
        // check crc
        uint8_t calc_crc = crc8(prefs_data.buf, PREFS_SIZE-1);
        
        if (calc_crc == prefs_data.prefs.crc)
            initialized = true;
    }
    
    if (!initialized) {        
        memset(prefs_data.buf, 0, PREFS_SIZE);
        prefs_data.prefs.sig0 = sig0;
        prefs_data.prefs.sig1 = sig1;
        prefs_data.prefs.disp_24h = 1;
        prefs_data.prefs.brightness = 10;
        prefs_data.prefs.dots = 1;
        prefs_data.prefs.gps = 1;
        prefs_data.prefs.gps_tzh = 9; // Tokyo, Japan
        prefs_data.prefs.gps_tzm = 0;
        prefs_data.prefs.auto_date = 1;
        prefs_data.prefs.flw = 0;
        prefs_data.prefs.crc = crc8(prefs_data.buf, PREFS_SIZE-1);
        
        IAP_Eeprom_Write(0, prefs_data.buf, PREFS_SIZE);
    }
}

void save_prefs()
{
    uint8_t calc_crc = crc8(prefs_data.buf, PREFS_SIZE-1);
    
    if (calc_crc != prefs_data.prefs.crc) {
        prefs_data.prefs.crc = calc_crc;
        IAP_Eeprom_Write(0, prefs_data.buf, PREFS_SIZE);
    }
}

PrefsData* get_prefs()
{
    if (prefs_data.prefs.flw == 2)
        flw.setCensored(false);
    else
        flw.setCensored(true); 

    return &prefs_data;    
}

void set_year(int year)
{
    s_pref_year = year;
}

void set_month(int month)
{
    s_pref_month = month;
}

void set_day(int day)
{
    s_pref_day = day;
}


void set_extra_prefs(int year, int month, int day)
{
    s_pref_year = year;
    s_pref_month = month;
    s_pref_day = day;
}

void get_extra_prefs(int& year, int& month, int& day)
{
    year = s_pref_year;
    month = s_pref_month;
    day = s_pref_day;
}

void set_pref(PREFS pref, uint32_t value)
{
    switch (pref) {
    case PREF_24H:
        prefs_data.prefs.disp_24h = value;
        break;
    case PREF_BRIGHTNESS:
        prefs_data.prefs.brightness = value;
        display.setBrightness(value);
        break;
    case PREF_ALARM:
        prefs_data.prefs.alarm = value;
        break;
    case PREF_YEAR:
        s_pref_year = value;
        break;
    case PREF_MONTH:
        s_pref_month = value;
        break;
    case PREF_DAY:
        s_pref_day = value;
        break;
    case PREF_DOTS:
        prefs_data.prefs.dots = value;
        break;
    case PREF_GPS:
        prefs_data.prefs.gps = value;
        break;
    case PREF_GPS_TZH:
        prefs_data.prefs.gps_tzh = value;
        break;
    case PREF_GPS_TZM:
        prefs_data.prefs.gps_tzm = value;
        break;
    case PREF_AUTODATE:
        prefs_data.prefs.auto_date = value;
        break;
    case PREF_FLW:
        prefs_data.prefs.flw = value;
        break;
    default:
        break;
    }        
}

