/* This is an example program which reads in the 3-axis acceleration, pressure, and temperature data from
 * a FRDM-FXS-MULTI sensor board. It then uses an MTSAS SocketModem Shield board to send the read data over
 * a cellular connection to Aeris AerCloud using an MQTT client protocol.
 */

#include "mbed.h"
#include "mtsas.h"
#include "PubSubClient.h"
#include "FXLS8471Q.h"
#include "MPL3115A2.h"

//
// PLEASE READ THIS!
//
// Example was created for the following hardware:
//   ST Nucleo F401RE   http://developer.mbed.org/platforms/ST-Nucleo-F401RE/
//   Freescale Multi-Sensor Shield https://developer.mbed.org/components/Freescale-Multi-Sensor-Shield/
//   MultiTech Socket Modem Shield: http://developer.mbed.org/components/Multi-Tech-SocketModem-Arduino-Shield-MT/
//   MultiTech MTSMC-H5 GSM Socket Modem http://www.multitech.com/models/92503252LF#gsc.tab=0
//
// To get the sample running, you'll need to fill in the following parameters below
//   Your cellular provider's APN: _APN
//   AerCloud API Key: _AERCLOUD_API_APIKEY
//   AerCloud Account ID: _AERCLOUD_ACCOUNT_ID
//   AerCloud Container: _ACERCLOUD_CONTAINER
//
// The AerCloud container needa a Data Model with the following schema:
//    accel_x  FLOAT
//    accel_y  FLOAT 
//    accel-Z  FLOAT
//    pressure FLOAT
//    temperature FLOAT
//
// To check if data has made it in the container, create a long polling subscription in AerPort and they use the following url in your browswer
//
// http://longpoll.aercloud.aeris.com/v1/1/containers/subscriptions/<subscrption_name>/notificationChannels/longPoll?apiKey=<api_key>
//
// You should see the something that looks like this in the browser:
// {"sclContentInstances":[{"sclId":"nucleo-0001","containerId":"Nucleo_Test","contentInstance":{"id":"a40c8e60-8248-11e4-8b38-0677f0dfdf5e","contentSize":90,"creationTime":1418420922950,"content":{"contentType":"application/json","contentTypeBinary":"{\"x\":0.005615,\"y\":-0.041260,\"z\":1.015137,\"pressure\":101098.500000,\"temperature\":25.125000}"}}},
//
//

char _AERCLOUD_API_KEY[] = "SET_YOUR_API_KEY";
char _AERCLOUD_ACCOUNT_ID[] = "SET_YOUR_ACCOUNT_NUMBER";
char _AERCLOUD_CONTAINER[] = "Nucleo_Test";
char _AERCLOUD_DEVICE_ID[] = "nucleo-0001";

char _APN[] = "SET_YOUR_CELLULAR_PROVIDERS_APN" ";

char _host[] = "mqtt.aercloud.aeris.com";
int _port = 1883;

#define MPL3115A2_I2C_ADDRESS (0x60<<1)

#define DATA_INTERVAL 30

void callback(char* topic, char* payload, unsigned int len) {
    logInfo("topic: [%s]\r\npayload: [%s]", topic, payload);
}

int main() {
    printf("Hello World\r\n");
    MTSLog::setLogLevel(MTSLog::TRACE_LEVEL);
    
    // for Nucleo boards
    MTSSerialFlowControl io(D8, D2, D3, D6);
    io.baud(115200);
    
    Cellular* radio = CellularFactory::create(&io);
    if (! radio) {
        logFatal("failed to create Cellular object - exiting");
        return 1;
    }
    
    radio->configureSignals(D4,D7,RESET);
    Transport::setTransport(radio);
    
    while (radio->setApn(_APN) != MTS_SUCCESS) {
        logError("failed to set APN [%s]", _APN);
        wait(2);
    }
    
    while (! radio->connect()) {
        logError("failed to bring up PPP link");
        wait(2);
    }
    
    printf("Signal Strength: %d\n\r", radio->getSignalStrength()); //Check the signal strength should be above 8
    
    // If you suspect a connectivity issue, uncomment the code below and if ping works.  If you are not getting a
    //  valid ping, there's a connectivity problem.  First step is to verify you've got the right APN set
    //
    // Try pinging default server "8.8.8.8" (Google's DNS)
    //int ping_valid = 0;
    //while (ping_valid == 0) {
    //    ping_valid = radio->ping();
    //    printf("Ping Valid: %s\n\r", ping_valid ? "true" : "false");
    //    
    //    if (ping_valid == 0) {
    //        wait(3);
    //    }
    //}
    
    PubSubClient mqtt(_host, _port, callback);
    
    char buf[128];
    
    FXLS8471Q acc(D11, D12, D13, D10);
    MPL3115A2 alt(D14, D15, MPL3115A2_I2C_ADDRESS, D4, D3);
    alt.Barometric_Mode(); 
    
    float acc_data[3];
    
    while (true) {
        if (! mqtt.connect(_AERCLOUD_DEVICE_ID, _AERCLOUD_ACCOUNT_ID, _AERCLOUD_API_KEY)) {
            logError("failed to connect to AerCloud Server");
            wait(5);
            continue;
        }
        
        acc.ReadXYZ(acc_data);
        
        snprintf(buf, sizeof(buf), "{\"x\":%f,\"y\":%f,\"z\":%f,\"pressure\":%f,\"temperature\":%f}", acc_data[0],acc_data[1],acc_data[2], alt.getPressure(), alt.getTemperature());
        logInfo("publishing: [%s]", buf);
        if (! mqtt.publish(_AERCLOUD_CONTAINER, buf)) {
            logError("failed to publish: [%s]", buf);
        }
        wait(1);
        mqtt.loop();
        mqtt.disconnect();
        wait(DATA_INTERVAL);
    }
    
}