/*
 *  AT&T IoT Starter Kit example using Amazon Web Service 
 */
#include "mbed.h"

// SD File System
#include "SDFileSystem.h"

// Serial extension
#include "MODSERIAL.h"

// Network includes
#include "WNCInterface.h"
#include "network_interface.h"

// AWS includes
#include "aws_iot_log.h"
#include "aws_iot_version.h"
#include "aws_iot_shadow_interface.h"
#include "aws_iot_shadow_json_data.h"
#include "aws_iot_config.h"
#include "aws_iot_mqtt_interface.h"

// Sensors
#include "HTS221.h"

#if DEBUG_LEVEL > 0
#include "mbedtls/debug.h"
#endif

//=====================================================================================================================
//
// Defines
//
//=====================================================================================================================
// LED Colors
#define COLOR_OFF    0x00
#define COLOR_RED    0x01
#define COLOR_GREEN  0x02
#define COLOR_BLUE   0x04
#define COLOR_WHITE  0x07
#define NUM_COLORS   5

// AWS defines
#define PATH_MAX    1024
#define MAX_LENGTH_OF_UPDATE_JSON_BUFFER 200 // NOTE: Be wary of this if your JSON doc grows
#define SHADOW_SYNC_INTERVAL 3.0             // How often we sync with AWS Shadow (in seconds)

// Comment out the following line if color is not supported on the terminal
//#define USE_COLOR
#ifdef USE_COLOR
 #define BLK "\033[30m"
 #define RED "\033[31m"
 #define GRN "\033[32m"
 #define YEL "\033[33m"
 #define BLU "\033[34m"
 #define MAG "\033[35m"
 #define CYN "\033[36m"
 #define WHT "\033[37m"
 #define DEF "\033[39m"
#else
 #define BLK
 #define RED
 #define GRN
 #define YEL
 #define BLU
 #define MAG
 #define CYN
 #define WHT
 #define DEF
#endif

// Sensor defines
#define CTOF(x)  ((x)*1.8+32) // Temperature

//=====================================================================================================================
//
// Globals
//
//=====================================================================================================================
// Controls LED color
unsigned char ledColor = COLOR_OFF;

// Color cycle array (used with SW3 button presses)
unsigned char colorCycle[NUM_COLORS] = {COLOR_OFF, COLOR_RED, COLOR_GREEN, COLOR_BLUE, COLOR_WHITE};

// Button interrupts
bool buttonOverride = false;
InterruptIn Interrupt(SW3);

// These defines are pulled from aws_iot_config.h
char HostAddress[255] = AWS_IOT_MQTT_HOST;
char MqttClientID[32] = AWS_IOT_MQTT_CLIENT_ID;
char ThingName[32] = AWS_IOT_MY_THING_NAME;
char PortString[5] = "8883";
uint32_t port = AWS_IOT_MQTT_PORT;
char iccidName[21] = "12345678901234567890";

// Sensor data
float temperature = 0.0;
int   humidity    = 0;

// Temp/humidity object
HTS221 hts221; 

//=====================================================================================================================
//
// Devices
//
//=====================================================================================================================
// GPIOs for RGB LED
DigitalOut led_green(LED_GREEN);
DigitalOut led_red(LED_RED);
DigitalOut led_blue(LED_BLUE);

// USB Serial port (to PC)
MODSERIAL pc(USBTX,USBRX,256,256);

// SD card access (MOSI, MISO, SCK, CS)
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd");

// I2C bus (SDA, SCL)
I2C i2c(PTC11, PTC10);

//=====================================================================================================================
//
// Functions
//
//=====================================================================================================================
//*********************************************************************************************************************
//* Prints the given format to the PC serial port.  Exposed to all files via aws_iot_log.h
//*********************************************************************************************************************
void pc_print(const char * format, ...)
{
    va_list vl;
    va_start(vl, format);
    pc.vprintf(format, vl);
    va_end(vl);
}

//*********************************************************************************************************************
//* Set the RGB LED's Color
//* LED Color 0=Off to 7=White.  3 bits represent BGR (bit0=Red, bit1=Green, bit2=Blue) 
//*********************************************************************************************************************
void SetLedColor(unsigned char ucColor)
{    
    //Note that when an LED is on, you write a 0 to it:
    led_red = !(ucColor & 0x1); //bit 0
    led_green = !(ucColor & 0x2); //bit 1
    led_blue = !(ucColor & 0x4); //bit 2
}

//*********************************************************************************************************************
//* SW3 Button handler.  Finds the current LED color and sets the button to the next color in colorCycle[]
//*********************************************************************************************************************
void sw3ButtonHandler()
{
    int i;
    for(i=0; i < NUM_COLORS; i++) {
        if (ledColor == colorCycle[i])
            break;
    }
    
    // (circular-queue)
    if (++i == NUM_COLORS)
        i = 0;
        
    ledColor = colorCycle[i];
    SetLedColor(ledColor);
    buttonOverride = true;
}

//*********************************************************************************************************************
//* Print LED and sensor data
//*********************************************************************************************************************
void printData()
{
    INFO("Temperature is: %0.2f F", temperature);
    INFO("Humidity    is: %02d", humidity);      
    switch (ledColor) {
         case COLOR_OFF:
             INFO("LED: Off");
             break;
         case COLOR_RED:
             INFO("LED: Red");
             break;
         case COLOR_GREEN:
             INFO("LED: Green");
             break;
         case COLOR_BLUE:
             INFO("LED: Blue");
             break;
         case COLOR_WHITE:
             INFO("LED: White");
             break;
    }
}

//=====================================================================================================================
//
// AWS Shadow Callbacks
//
//=====================================================================================================================
//*********************************************************************************************************************
//* This is the callback function that fires when an update is sent.  It will print the update response status.
//*********************************************************************************************************************
void ShadowUpdateStatusCallback(const char *pThingName, ShadowActions_t action, Shadow_Ack_Status_t status,
        const char *pReceivedJsonDocument, void *pContextData) {

    INFO("Shadow Update Status Callback");
    
    if (status == SHADOW_ACK_TIMEOUT) {
        INFO("Update Timeout--");
    } else if (status == SHADOW_ACK_REJECTED) {
        INFO("Update RejectedXX");
    } else if (status == SHADOW_ACK_ACCEPTED) {
        INFO("Update Accepted!!"); // Good
    }
}

//*********************************************************************************************************************
//* This is the callback function that fires when AWS has sends out a shadow update. 
//*********************************************************************************************************************
void ledControl_Callback(const char *pJsonString, uint32_t JsonStringDataLen, jsonStruct_t *pContext) {
    
    INFO("LED Callback Detected.");
    
    if (pContext != NULL) {
        switch (*(unsigned char *)(pContext->pData)){   
            case COLOR_OFF:
                INFO("LED -> OFF (%d)", *(unsigned char *)(pContext->pData));
                break;
            case COLOR_RED:
                INFO("LED -> RED (%d)", *(unsigned char *)(pContext->pData));
                break;
            case COLOR_GREEN:
                INFO("LED -> GREEN (%d)", *(unsigned char *)(pContext->pData));
                break;
            case COLOR_BLUE:
                INFO("LED -> BLUE (%d)", *(unsigned char *)(pContext->pData));
                break;
            case COLOR_WHITE:
                INFO("LED -> WHITE (%d)", *(unsigned char *)(pContext->pData));
                break;    
        }
    }
    else {
        INFO("pContext was detected as NULL");
    }
}
 
//*********************************************************************************************************************
//* Subscribe callback (used with alternate demo)
//*********************************************************************************************************************
int MQTTcallbackHandler(MQTTCallbackParams params) {

    INFO("Subscribe callback");
    INFO("%.*s\t%.*s",
            (int)params.TopicNameLen, params.pTopicName,
            (int)params.MessageParams.PayloadLen, (char*)params.MessageParams.pPayload);

    return 0;
}

//*********************************************************************************************************************
//* Disconnect handling (used with alternate demo)
//*********************************************************************************************************************
void disconnectCallbackHandler(void) {
    WARN("MQTT Disconnect");
    IoT_Error_t rc = NONE_ERROR;
    if(aws_iot_is_autoreconnect_enabled()){
        INFO("Auto Reconnect is enabled, Reconnecting attempt will start now");
    }else{
        WARN("Auto Reconnect not enabled. Starting manual reconnect...");
        rc = aws_iot_mqtt_attempt_reconnect();
        if(RECONNECT_SUCCESSFUL == rc){
            WARN("Manual Reconnect Successful");
        }else{
            WARN("Manual Reconnect Failed - %d", rc);
        }
    }
}

//=====================================================================================================================
//
// Out-of-Box Demo: This function is used as part of the binary that comes with the Starter Kit.  Instead of using an
//                  AWS device shadow, it publishes to an AWS Rule. The Rule is setup to store data to a DynamoDB, and
//                  the demo S3 website pulls that data from the DynamoDB and displays it.
//
//=====================================================================================================================
int outOfBoxDemo() {
    INFO("Running Out-of-Box Function (alternate demo).");
    
    IoT_Error_t rc = NONE_ERROR;
    int32_t i = 0;
    int publishCount = 0;
    bool infinitePublishFlag = true;
    char cPayload[100];
    char cTopic[100];
    const string colorStrings[] = {"Off", "Red", "Green", "", "Blue", "", "", "White"};
    float updateInterval = 1.0; // seconds

    MQTTConnectParams connectParams = MQTTConnectParamsDefault;
    connectParams.KeepAliveInterval_sec = 10;
    connectParams.isCleansession = true;
    connectParams.MQTTVersion = MQTT_3_1_1;
    connectParams.pClientID = iccidName;  // Using ICCID for unique Client ID
    connectParams.pHostURL = HostAddress;
    connectParams.port = port;
    connectParams.isWillMsgPresent = false;
    connectParams.pRootCALocation = AWS_IOT_ROOT_CA_FILENAME;
    connectParams.pDeviceCertLocation = AWS_IOT_CERTIFICATE_FILENAME;
    connectParams.pDevicePrivateKeyLocation = AWS_IOT_PRIVATE_KEY_FILENAME;
    connectParams.mqttCommandTimeout_ms = 10000;
    connectParams.tlsHandshakeTimeout_ms = 10000;
    connectParams.isSSLHostnameVerify = true; // ensure this is set to true for production
    connectParams.disconnectHandler = disconnectCallbackHandler;

    INFO("Connecting...");
    rc = aws_iot_mqtt_connect(&connectParams);
    if (NONE_ERROR != rc) {
        ERROR("Error(%d) connecting to %s:%d", rc, connectParams.pHostURL, connectParams.port);
    }
    
    /*
     * Enable Auto Reconnect functionality. Minimum and Maximum time of Exponential backoff are set in aws_iot_config.h
     *  #AWS_IOT_MQTT_MIN_RECONNECT_WAIT_INTERVAL
     *  #AWS_IOT_MQTT_MAX_RECONNECT_WAIT_INTERVAL
     */
    INFO("Set Auto Reconnect...");
    rc = aws_iot_mqtt_autoreconnect_set_status(true);
    if (NONE_ERROR != rc) {
        ERROR("Unable to set Auto Reconnect to true - %d", rc);
        return rc;
    }

    // Comment this in if you want to subscribe
    /*MQTTSubscribeParams subParams = MQTTSubscribeParamsDefault;
    subParams.mHandler = MQTTcallbackHandler;
    subParams.pTopic = "sdkTest/sub";
    subParams.qos = QOS_0;

    if (NONE_ERROR == rc) {
        INFO("Subscribing...");
        rc = aws_iot_mqtt_subscribe(&subParams);
        if (NONE_ERROR != rc) {
            ERROR("Error subscribing");
        }
    }*/

    // Initializ the payload
    MQTTMessageParams Msg = MQTTMessageParamsDefault;
    Msg.qos = QOS_0;
    Msg.pPayload = (void *) cPayload;

    MQTTPublishParams Params = MQTTPublishParamsDefault;
    
    // Sets up the topic to publish to
    sprintf(cTopic, AWS_IOT_MY_TOPIC, iccidName);
    Params.pTopic = cTopic;

    if (publishCount != 0) {
        infinitePublishFlag = false;
    }
      
    INFO("READY TO PUBLISH! Press SW3 button to publish current data.");
    while ((NETWORK_ATTEMPTING_RECONNECT == rc || RECONNECT_SUCCESSFUL == rc || NONE_ERROR == rc)
            && (publishCount > 0 || infinitePublishFlag)) {

        // Max time the yield function will wait for read messages
        rc = aws_iot_mqtt_yield(100);
        if(NETWORK_ATTEMPTING_RECONNECT == rc){
            INFO("--> sleep (attempting to reconnect)");
            wait(1);
            // If the client is attempting to reconnect we will skip the rest of the loop.
            continue;
        }
        
        // Whenever the software button (SW3) is pressed the LED will changes color and this will
        // trigger a publish to the AWS topic specified.
        if (buttonOverride) {
            buttonOverride = false;
            
            // Get temp/humidity values
            temperature = CTOF(hts221.readTemperature());
            humidity = hts221.readHumidity();
    
            // Loading data into JSON format
            sprintf(cPayload, "{\"color\":\"%s\",\"temperature\":%f,\"humidity\":%d}", colorStrings[ledColor], temperature, humidity);
            Msg.PayloadLen = strlen(cPayload) + 1;
            Params.MessageParams = Msg;
            
            // Publish
            rc = aws_iot_mqtt_publish(&Params);
            if (publishCount > 0) {
                publishCount--;
            }
                      
            printData();
            INFO("--> Update sent. Sleep for %f seconds", updateInterval);
            wait(updateInterval-.02);
        }
        else {
            wait(.3); // 300 ms
        }
    }

    if (NONE_ERROR != rc) {
        ERROR("An error occurred in the loop.\n");
    } else {
        INFO("Publish done\n");
    }

    return rc;
}

//=====================================================================================================================
//
// Main
//
//=====================================================================================================================
int main() {
    
    // Set baud rate for PC Serial
    pc.baud(115200);
    INFO("Hello World from AT&T IoT Start Kit demo!");
              
    int i;          
    IoT_Error_t rc = NONE_ERROR;  
    char JsonDocumentBuffer[MAX_LENGTH_OF_UPDATE_JSON_BUFFER];
    size_t sizeOfJsonDocumentBuffer = sizeof(JsonDocumentBuffer) / sizeof(JsonDocumentBuffer[0]);

    // JSON struct for LED control
    jsonStruct_t ledController;
    ledController.cb = ledControl_Callback;
    ledController.pData = &ledColor;
    ledController.pKey = "ledColor";
    ledController.type = SHADOW_JSON_UINT8;

    // JSON struct for temperature\humidity readings
    jsonStruct_t temperatureHandler;
    temperatureHandler.cb = NULL;
    temperatureHandler.pKey = "temperature";
    temperatureHandler.pData = &temperature;
    temperatureHandler.type = SHADOW_JSON_FLOAT;
    
    jsonStruct_t humidityHandler;
    humidityHandler.cb = NULL;
    humidityHandler.pKey = "humidity";
    humidityHandler.pData = &humidity;
    humidityHandler.type = SHADOW_JSON_INT16;
    
    INFO("AWS IoT SDK Version(dev) %d.%d.%d-%s", VERSION_MAJOR, VERSION_MINOR, VERSION_PATCH, VERSION_TAG);

#ifdef USING_SD_CARD
    // Paths for certs from SD card
    INFO("Using SD card files for AWS config.");  
    DEBUG("- mqtt config path: %s", AWS_MQTT_CONFIG_FILENAME);
    DEBUG("- rootCA path: %s", AWS_IOT_ROOT_CA_FILENAME);
    DEBUG("- clientCRT path: %s", AWS_IOT_CERTIFICATE_FILENAME);
    DEBUG("- clientKey path: %s", AWS_IOT_PRIVATE_KEY_FILENAME);
#else
    INFO("Using #defines in aws_iot_config.h and certs from certs.cpp for AWS config.");      
#endif
    
    // Startup signal - blinks through RGBW then turns off
    SetLedColor(COLOR_RED);
    wait(.5);
    SetLedColor(COLOR_GREEN);
    wait(.5);
    SetLedColor(COLOR_BLUE);
    wait(.5);
    SetLedColor(COLOR_WHITE);
    wait(.5);
    SetLedColor(COLOR_OFF);
    
    // Initialize sensors
    INFO("Init sensors...");
    void hts221_init(void);
    i = hts221.begin();  
    if(!i) {
        WARN(RED "HTS221 NOT DETECTED!!\n\r");
    }
      
    // Setup SW3 button to falling edge interrupt
    INFO("Init interrupts...");
    Interrupt.fall(&sw3ButtonHandler);
      
    // Boot the Avnet Shield before any other operations
    INFO("Net Boot...");
    net_modem_boot();
    
    //==========================================================================
    // NOTE:  You can comment in the following line for an alternate demo that
    // is used as the out-of-box demo binary that comes with the  AT&T IoT 
    // Starter Kit.  It loops instead of the rest of Main()
    //return outOfBoxDemo();
    //==========================================================================
          
    // Intialize MQTT/Cert parameters
    ShadowParameters_t sp = ShadowParametersDefault;
#ifdef USING_SD_CARD
    rc = (IoT_Error_t)mbedtls_mqtt_config_parse_file(&sp, AWS_MQTT_CONFIG_FILENAME);
    if (NONE_ERROR != rc) {
        ERROR("Failed to initialize mqtt parameters %d", rc);
        return rc;
    }   
    sp.pClientCRT = AWS_IOT_CERTIFICATE_FILENAME;
    sp.pClientKey = AWS_IOT_PRIVATE_KEY_FILENAME;
    sp.pRootCA = AWS_IOT_ROOT_CA_FILENAME;
#else
    sp.pMyThingName = AWS_IOT_MY_THING_NAME;
    sp.pMqttClientId = AWS_IOT_MQTT_CLIENT_ID;
    sp.pHost = HostAddress;
    sp.port = port;
    
    sp.pClientCRT = AWS_IOT_CERTIFICATE_FILENAME;
    sp.pClientKey = AWS_IOT_PRIVATE_KEY_FILENAME;
    sp.pRootCA = AWS_IOT_ROOT_CA_FILENAME;
#endif
          
    INFO("Initialize the MQTT client...");
    MQTTClient_t mqttClient;
    aws_iot_mqtt_init(&mqttClient);

    INFO("Shadow Init...");
    rc = aws_iot_shadow_init(&mqttClient);
    if (NONE_ERROR != rc) {
        ERROR("Shadow Init Error %d", rc);
        return rc;
    }
    
    INFO("Shadow Connect...");   
    rc = aws_iot_shadow_connect(&mqttClient, &sp);
    if (NONE_ERROR != rc) {
        ERROR("Shadow Connection Error %d", rc);
        return rc;
    }

    // Enable Auto Reconnect functionality. Minimum and Maximum time of Exponential backoff are set in aws_iot_config.h
    // #AWS_IOT_MQTT_MIN_RECONNECT_WAIT_INTERVAL
    // #AWS_IOT_MQTT_MAX_RECONNECT_WAIT_INTERVAL
    rc = mqttClient.setAutoReconnectStatus(true);
    if (NONE_ERROR != rc) {
        ERROR("Unable to set Auto Reconnect to true - %d", rc);
        return rc;
    }
    
    // Example line of how to delete a shadow (not used in this demo)
    //aws_iot_shadow_delete(&mqttClient, AWS_IOT_MY_THING_NAME, ShadowUpdateStatusCallback, NULL, 8, true);

    INFO("Shadow Register Delta...");
    rc = aws_iot_shadow_register_delta(&mqttClient, &ledController);
    if (NONE_ERROR != rc) {
        ERROR("Shadow Register Delta Error");
        return rc;
    }
    
    INFO("Will attempt to sync with device shadow every %f seconds.", SHADOW_SYNC_INTERVAL);
    // Loop and publish changes from the FRDM board
    while (NETWORK_ATTEMPTING_RECONNECT == rc || RECONNECT_SUCCESSFUL == rc || NONE_ERROR == rc) {
        
        // Looks for incoming socket messages
        rc = aws_iot_shadow_yield(&mqttClient, 200);
        if (NETWORK_ATTEMPTING_RECONNECT == rc) {
            // If the client is attempting to reconnect we will skip the rest of the loop.
            INFO("Attempting to reconnect...");
            wait(1);
            continue;
        }
        
        // Read sensor data
        temperature = CTOF(hts221.readTemperature());
        humidity = hts221.readHumidity();
                                
        INFO("\n=======================================================================================\n");             
        // Initialize JSON shadow document          
        rc = aws_iot_shadow_init_json_document(JsonDocumentBuffer, sizeOfJsonDocumentBuffer);
        if (rc == NONE_ERROR) {
            
            // If there has been a SW3 button press update the 'desired' color
            if (buttonOverride) {
                rc = aws_iot_shadow_add_desired(JsonDocumentBuffer, sizeOfJsonDocumentBuffer, 1, &ledController);               
                buttonOverride = false;
            }
                  
            // Updates the 'reported' color/temp/humidity
            rc = aws_iot_shadow_add_reported(JsonDocumentBuffer, sizeOfJsonDocumentBuffer, 3, &ledController,
                                                                                              &temperatureHandler,
                                                                                              &humidityHandler);
                 
            if (rc == NONE_ERROR) {               
                rc = aws_iot_finalize_json_document(JsonDocumentBuffer, sizeOfJsonDocumentBuffer);   
                            
                if (rc == NONE_ERROR) {
                    INFO("Update Shadow: %s", JsonDocumentBuffer);
                    rc = aws_iot_shadow_update(&mqttClient, sp.pMyThingName, JsonDocumentBuffer,
                            ShadowUpdateStatusCallback, NULL, 8, true);
                }
            }
        }  
        
        // Print data    
        printData();
        INFO("*****************************************************************************************");
         
        // Set the LED color    
        SetLedColor(ledColor);
        wait(SHADOW_SYNC_INTERVAL);
    }

    if (NONE_ERROR != rc) {
        ERROR("An error occurred in the loop %d", rc);
    }

    INFO("Disconnecting");
    rc = aws_iot_shadow_disconnect(&mqttClient);

    if (NONE_ERROR != rc) {
        ERROR("Disconnect error %d", rc);
    }

    return rc;   
}