#include "mbed.h"
#include "XNucleo53L0A1.h"
#include "tof_gestures_DIRSWIPE_1.h"
#include "tof_gestures.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

/*
 * This VL53L0X Expansion board sample application performs range measurements using
 * range_continuous_interrupt mode to generate a hardware interrupt each time a new
 * measurement is ready to be read.
 * The application supports the centre, on-board, sensor and up two satellite boards.
 *
 * The measured range data is displayed on the on-board 4-digit LED display.
 *
 * The User Blue button switches between the currently selected sensor to display range
 * results from.
 *
 * The Black Reset button is used to restart the program.
 *
 * *** NOTE : By default hardlinks U10, U11, U15 & U18, on the underside of
 *            the X-NUCELO-53L0A1 expansion board are not made/OFF.
 *            These links must be made to allow interrupts from the Satellite boards
 *            to be received.
 *            U11 and U18 must be made/ON to allow interrupts to be received from the
 *            INT_L & INT_R positions; or
 *            U10 and U15 must be made/ON to allow interrupts to be received from the
 *            Alternate INT_L & INT_R positions.
 *            The X_NUCLEO_53L0A1 firmware library defaults to use the INT_L/INT_R
 *            positions.
 *            INT_L is available on expansion board Arduino Connector CN5, pin 1 as D8.
 *            Alternate INT_L is on CN5 Connector pin 2 as D9.
 *            INT_R is available on expansion board Arduino Connector CN9, pin 3 as D2.
 *            Alternate INT_R is on CN9 Connector pin 5 as D4.
 *            The pinouts are shown here : https://developer.mbed.org/components/X-NUCLEO-53L0A1/
 *
 */

#define VL53L0_I2C_SDA   D14
#define VL53L0_I2C_SCL   D15

#if USER_BUTTON==PC_13  // we are cross compiling for Nucleo-64s
InterruptIn stop_button(USER_BUTTON);
#endif

static XNucleo53L0A1 *board = NULL;
VL53L0X_RangingMeasurementData_t data_sensor;
VL53L0X_RangingMeasurementData_t data_sensor_l;
VL53L0X_RangingMeasurementData_t data_sensor_r;
OperatingMode operating_mode;

Gesture_DIRSWIPE_1_Data_t gestureDirSwipeData;

/* current displayed sensor change IRQ */
volatile bool switchChanged = false;

/* interrupt requests */
volatile bool centerSensor = false;
volatile bool leftSensor = false;
volatile bool rightSensor = false;

/* Current sensor number*/
volatile int currentSensor = 0;
/* Installed sensors count */
int sensorCnt = 0;

/* installed sensors prefixes */
char installedSensors[3];

/* ISR callback function of the sensor_centre */
void sensor_centre_irq(void)
{
    centerSensor = true;
    board->sensor_centre->disable_interrupt_measure_detection_irq();
}

void sensor_left_irq(void)
{
    leftSensor = true;
    board->sensor_left->disable_interrupt_measure_detection_irq();
}

void sensor_right_irq(void)
{
    rightSensor = true;
    board->sensor_right->disable_interrupt_measure_detection_irq();
}

/* ISR callback function of the user blue button to switch measuring sensor. */
void switch_measuring_sensor_irq(void)
{
    stop_button.disable_irq();
    switchChanged = true;
}


/* On board 4 digit local display refresh */
void refresh_display(const VL53L0X_RangingMeasurementData_t &data, char prefix)
{
    static char str[5];
    if (data_sensor.RangeStatus == 0) { // we have a valid range.
        sprintf(str, "%c%3d", prefix, data.RangeMilliMeter);
        board->display->display_string(str);
    } else {
        sprintf(str, "%c%s", prefix, "---");
        board->display->display_string(str);
    }
}

DigitalOut sl(PC_10);
DigitalOut sr(PA_8);

inline void measure_sensors(OperatingMode op_mode)
{
    bool current = false;
    int32_t leftRange, rightRange;
    int gesture_code;
    int meas_ready_l, meas_ready_r;
    uint8_t measurement_data_ready=0;   
    VL53L0X_Error status;     
       
    meas_ready_r =0; meas_ready_l =0;
    data_sensor_l.RangeMilliMeter = 1200;
    data_sensor_r.RangeMilliMeter = 1200;        
    leftRange=1200;
    rightRange=1200;
    VL53L0X_DEV dev_l, dev_r;
    VL53L0X_RangingMeasurementData_t ranging_measurement_data_l;
    VL53L0X_RangingMeasurementData_t ranging_measurement_data_r;    
    
    board->sensor_left->vl53l0x_get_device(&dev_l);   
    board->sensor_right->vl53l0x_get_device(&dev_r);  

    sl=0;
    sr=0;   
    // start single range measure
    status = board->sensor_left->VL53L0X_start_measurement(dev_l);
    if (status != 0) printf ("ERROR line: %d\n\r",__LINE__);    
    status = board->sensor_right->VL53L0X_start_measurement(dev_r);    
    if (status != 0) printf ("ERROR line: %d\n\r",__LINE__);        
    
    for (;meas_ready_r==0 && meas_ready_l==0;) {
    
        status = board->sensor_left->VL53L0X_get_measurement_data_ready(dev_l, &measurement_data_ready);
        if (status != 0) printf ("ERROR line: %d\n\r",__LINE__);           
        if (measurement_data_ready ==1 && status ==0 ) {
            sl=1;            
            board->sensor_left->VL53L0X_get_ranging_measurement_data(dev_l, &ranging_measurement_data_l);  
            board->sensor_left->VL53L0X_clear_interrupt_mask(dev_l, 0);                    
//            printf ("L = %d\n\r", ranging_measurement_data_l.RangeMilliMeter);
            meas_ready_l = 1;
            leftRange = ranging_measurement_data_l.RangeMilliMeter;  
            sl=0;
        } else {
            meas_ready_l = 0;
            leftRange = 1200;
            ranging_measurement_data_l.RangeMilliMeter = 1200;
        }
    
        status = board->sensor_right->VL53L0X_get_measurement_data_ready(dev_r, &measurement_data_ready);
        if (status != 0) printf ("ERROR line: %d\n\r",__LINE__);           
        if (measurement_data_ready ==1 && status ==0 ) {
            sr=1;            
            board->sensor_right->VL53L0X_get_ranging_measurement_data(dev_r, &ranging_measurement_data_r); 
            board->sensor_right->VL53L0X_clear_interrupt_mask(dev_r, 0);                                
//            printf ("R = %d\n\r", ranging_measurement_data_r.RangeMilliMeter);
            meas_ready_r = 1;
            rightRange = ranging_measurement_data_r.RangeMilliMeter;           
            sr=0;  
        } else {
           meas_ready_r = 0;
           rightRange = 1200;
           data_sensor_r.RangeMilliMeter = 1200;
        }        
    }
               
    gesture_code = tof_gestures_detectDIRSWIPE_1(leftRange, rightRange, &gestureDirSwipeData);

    /* Format data to display */
    if(gesture_code == GESTURES_SWIPE_LEFT_RIGHT){
        printf ("right --->>>\n\r");
    }else if(gesture_code == GESTURES_SWIPE_RIGHT_LEFT){
        printf ("<<<---\n\r");                    
    }else{
        //ShowGestureHelpMsg(gesture_code);
    }    
}

int init_sensors_array()
{
    int status = 1;
    sensorCnt = 0;
    /* start the measure on the center sensor */
    if (NULL != board->sensor_centre) {
        installedSensors[sensorCnt] = 'C';
        status = board->sensor_centre->stop_measurement(operating_mode);
//        status = board->sensor_centre->start_measurement(operating_mode, &sensor_centre_irq);
        ++sensorCnt;
    }
    /* start the measure on the left sensor */
    if (NULL != board->sensor_left) {
        installedSensors[sensorCnt] = 'L';        
        VL53L0X_DEV dev_l;
        board->sensor_left->vl53l0x_get_device(&dev_l);            
        uint8_t vhv_settings, phase_cal;
        board->sensor_left->VL53L0X_perform_ref_calibration(dev_l, &vhv_settings, &phase_cal);
        board->sensor_left->VL53L0X_set_measurement_timing_budget_micro_seconds(dev_l, 20000);
        ++sensorCnt;
    }
    
    /* start the measure on the right sensor */
    if (NULL != board->sensor_right) {
        installedSensors[sensorCnt] = 'R';
        VL53L0X_DEV dev_r;
        board->sensor_right->vl53l0x_get_device(&dev_r);            
        uint8_t vhv_settings, phase_cal;
        board->sensor_right->VL53L0X_perform_ref_calibration(dev_r, &vhv_settings, &phase_cal);
        board->sensor_right->VL53L0X_set_measurement_timing_budget_micro_seconds(dev_r, 20000);
        ++sensorCnt;
    }
    currentSensor = 0;
    return status;
}


void range_measure(DevI2C *device_i2c)
{
    int status;

    /* creates the 53L0A1 expansion board singleton obj */
    board = XNucleo53L0A1::instance(device_i2c, A2, D8, D2);
    //board=X_NUCLEO_53L0A1::instance(device_i2c, A2, D9, D4); // Alternate INT_L/INT_R settings.

    board->display->display_string("53L0");

//    operating_mode = range_continuous_interrupt;
    operating_mode = range_single_shot_polling;

    /* init the 53L0A1 expansion board with default values */
    status = board->init_board();

    if (status) {
        printf("Failed to init board!\n\r");
    } else {
        status = init_sensors_array();
    }
  /* Initialize directional swipes recognition : swipe detected below 400 mm, no max speed, min duration is 1 sec for a swipe and hand must not cover both devices */
    tof_gestures_initDIRSWIPE_1(400, 0, 1000, false, &gestureDirSwipeData);
  /* Select which module to debug (code must be compiled with TRACE defined in compiler command line) */
  TOF_GESTURES_DEBUG_SET_MODULES(NONE); // Could be NONE or TAP_1|TAP_SWIPE_2|DIRSWIPE_1 for instance (or any other combinations);
  
    if (!status) {
        printf("\r\nEntering loop mode\r\n");
        while (true) {
            measure_sensors(operating_mode);
            if (switchChanged) {
                ++currentSensor;
                if (currentSensor == sensorCnt)
                    currentSensor = 0;

                printf("Sensor changed to %c\r\n", installedSensors[currentSensor]);
                switchChanged = false;
                stop_button.enable_irq();
            }            
        }
    }
    delete board;
}

/*=================================== Main ==================================
 Press the blue user button to switch the displayed sensor.
=============================================================================*/
int main()
{
#if USER_BUTTON==PC_13  // we are cross compiling for Nucleo-f401 
    stop_button.rise(&switch_measuring_sensor_irq);
    stop_button.enable_irq();
#endif
    DevI2C *device_i2c = new DevI2C(VL53L0_I2C_SDA, VL53L0_I2C_SCL);
    range_measure(device_i2c);  // start continuous measures
}
