/** Implements Sensor Control for CES Instrumented Booth */

#include "mbed.h"
#include "sensor_ctl.h"
#include "node_cfg.h"

//Sensor Drivers
#include "htu21d.h"
#include "MAX9814.h"
#include "sonar.h"
#include "Presence.h"
#include "SHARPIR.h"
#include "DoorTrip.h"
//#include "FXOS8700Q.h"

//Sensor MDS Resources
#include "door_trip.h"
#include "height.h"
#include "presence_resource.h"
#include "sound_level.h"
#include "temperature.h"
#include "accelerometer.h"


//Common Sensors
#if NODE_SENSOR_STATION
htu21d temphumid(PTE25, PTE24); //Temp humid sensor || SDA, SCL
MAX9814 microphone(PTB3); //Analogue in required.
#if NODE_PIR_STATION
presence pir(PTB11, false, PIR_SENSOR_DEBOUNCE_MS); //(InterruptPin), for PIR sensor, 
#endif //NODE PIR STATION
#if NODE_KIOSK_STATION
presence kiosk(PTB10, true, KIOSK_SENSOR_DEBOUNCE_MS); //(Interrupt pinrequired, no timeout)
#endif //NODE KIOSK STATION
#if NODE_DOOR_TRIP_STATION
DoorTrip laser(PTC11, true, DOOR_SENSOR_DEBOUNCE_MS ); //(AnalogIn required), for IR door trip
#endif //NODE TRIP STATION
#if NODE_HEIGHT_STATION
Timer sonarTimer;
Timer debounceHeight;
Sonar sonar(PTC10, sonarTimer);     //(AnalogIn required, Leave as SW2.)

#define DOOR_HEIGHT_START_MEASURING_THRESHOLD_CM    150
#define DOOR_HEIGHT_STOP_MEASURING_THRESHOLD_CM     140
#define DOOR_HEIGHT_STOP_MEASURING_SAMPLE_COUNT       2
#define DOOR_HEIGHT_SENSOR_MOUNT_HEIGHT_CM          220
#endif //NODE HEIGHT STATION
#if NODE_ACCELEROMETER_STATION 
FXOS8700Q_acc acc( PTE25, PTE24, FXOS8700CQ_SLAVE_ADDR1);     
#endif //NODE ACCELEROMETER STATION
#endif //NODE_SENSOR_STATION


//Variables provided to rest of applications
float  current_temperature_value = 0;
float  current_ambient_noise_value = 0;
float  current_ambient_noise_sum = 0;
int    current_ambient_noise_count = 0;
//Either height XOR kiosk presence XOR PIR station...
float    current_height_value = 0;
bool     current_presence_value = false;         //Either from Kiosk or PIR
//And it might have a door trip..
bool     current_door_trip_value = false;
bool     current_accelerometer_value = false;

//Door trip...
float door_trip_starting_volts = 0;
float maxValue=0;
bool set=0;  

//Door Height...
float door_height_max_value = 0;
bool door_height_measuring = 0;
int door_height_sample_count = 0;
    
//Initialisation
void init_sensors() {
//    #if NODE_DOOR_TRIP_STATION
//    door_trip_starting_volts = sharpir.volt();
//    #endif
    #if NODE_PIR_STATION
    wait(2); //Wait 2 seconds for sensor calibration period.
    #endif //PIR STATION
    #if NODE_ACCELEROMETER_STATION
    acc.enable();
    #endif
    #if NODE_HEIGHT_STATION
    sonarTimer.start();
    #endif
}

#if NODE_SENSOR_STATION

//timer handler functions
void handle_temperature_report_timer() {
        current_temperature_value = temphumid.getTemp();
        printf("Temperature Sample: %2.2f\r\n", current_temperature_value);
        temperature_report();
}

void handle_microphone_sample_timer(){
    float sample = microphone.sound_level();
    //printf("Sound Sample: %2.2f\r\n", sample);
    //Keep running sum and sample count to compute average...
    current_ambient_noise_sum += sample;
    current_ambient_noise_count++;
    current_ambient_noise_value = current_ambient_noise_sum / current_ambient_noise_count;
//    if (sample > current_ambient_noise_value){
//        current_ambient_noise_value = sample;
//    }
}

void handle_microphone_report_timer(){
    //Report.
    sound_level_report();
    //Reset noise...
    current_ambient_noise_value = 0;
    current_ambient_noise_sum = 0;
    current_ambient_noise_count = 0;
}

#if NODE_PIR_STATION
void handle_motion_report_timer(){
    bool new_pir = pir.isPresent();
    //printf("PIR Sample: %d\r\n", new_pir);
    //printf("Old PIR Sample: %d\r\n", current_presence_value);
    if(new_pir != current_presence_value) {
        //printf("Reporting PIR...\r\n");
        current_presence_value = new_pir;
        presence_report();
    }
}
#endif //NODE_PIR_STATION

#if NODE_KIOSK_STATION
void handle_kiosk_report_timer(){
    bool new_kiosk = kiosk.isPresent();
    if(new_kiosk != current_presence_value) {
        current_presence_value = new_kiosk;
        presence_report();
    }
}
#endif //NODE_KIOSK_STATION

#if NODE_DOOR_TRIP_STATION
void handle_door_trip_report_timer(){
    bool new_laser = laser.isPresent();

    if(new_laser != current_door_trip_value) {
        current_door_trip_value = new_laser;
        door_trip_report();
    }
}
#endif //NODE_DOOR_TRIP_STATION

#if NODE_HEIGHT_STATION
void handle_door_height_sample_timer(){
    int height_sample = DOOR_HEIGHT_SENSOR_MOUNT_HEIGHT_CM-sonar.read();
    if(height_sample > DOOR_HEIGHT_START_MEASURING_THRESHOLD_CM) {
        door_height_sample_count=0;
        if (height_sample > door_height_max_value) {
            door_height_max_value = height_sample;
        }
        door_height_measuring=true;
    }

    if((height_sample < DOOR_HEIGHT_STOP_MEASURING_THRESHOLD_CM && door_height_measuring )){
            door_height_sample_count++;
            if (door_height_sample_count > DOOR_HEIGHT_STOP_MEASURING_SAMPLE_COUNT){
            current_height_value=door_height_max_value;
            door_height_max_value=0,door_height_measuring=false;
            door_height_sample_count = 0;
            height_report();  
        }
    }
}
#endif //NODE HEIGHT STATION

#if NODE_ACCELEROMETER_STATION
void handle_accelerometer_report_timer(){
    bool new_accelerometer = acc.get_values();
    if(new_accelerometer != current_accelerometer_value) {

    current_accelerometer_value = new_accelerometer;
    accelerometer_report();
    }

}
#endif //NODE_ACCELEROMETER_STATION

#endif //NODE_SENSOR_STATION