/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
#include "EthernetInterface.h"
#include "nsdl_support.h"
#include "node_cfg.h"
#include "sensor_ctl.h"
#include "rtos.h"

#include "Watchdog.h"

Serial pc(PTC17,PTC16);
#define WATCHDOG_TIMEOUT_SEC 30 //30 seconds
#define HEARTBEAT_PERIOD_MS 1000 //1 second...

// Ethernet initialization
EthernetInterface eth;
static void ethernet_init()
{
    /* Initialize network */
#ifdef DHCP
    pc.printf("DHCP in use\r\n");
    eth.init();
#else
    eth.init(IP, MASK, GW);
#endif
    if(eth.connect(30000) == 0)
        pc.printf("Connect OK\n\r");
    pc.printf("IP Address:%s ", eth.getIPAddress());
}


//Hard Fault Handler (Watchdog)
extern "C" void HardFault_Handler() {
    pc.printf("Hard Fault!\r\n");
    NVIC_SystemReset();
}

// ****************************************************************************
// Socket initialization
UDPSocket server;
Endpoint nsp;


/* The number of seconds between NSDL Ticks*/
#define NSDL_TICK_PERIOD_MS  1000

void main_event_loop() {    
    //For timing control
//    Timer nsdlTickTimer;
    Timer registrationTimer;
    #if HEART_BEAT
    Timer heartBeatTimer;
    #endif
    
    //Sensor timers...
    #if NODE_SENSOR_STATION
    Timer temperatureReportTimer;
    Timer microphoneSampleTimer;
    Timer microphoneReportTimer;
    #if NODE_PIR_STATION
    Timer motionReportTimer;
    #endif //NODE PIR STATION
    #if NODE_KIOSK_STATION
    Timer kioskReportTimer;
    #endif //NODE KIOSK STATION
    #if NODE_DOOR_TRIP_STATION
    Timer doorTripReportTimer;
    #endif //NODE TRIP STATION
    #if NODE_HEIGHT_STATION
    Timer heightReportTimer;
    #endif //NODE HEIGHT STATION
    #endif //NODE_SENSOR_STATION
       
    //For recieving NSP messages
    sn_nsdl_addr_s received_packet_address;
    uint8_t received_address[4];
    int8_t nsdl_result = 0;
    char buffer[256];
    Endpoint from;
    memset(&received_packet_address, 0, sizeof(sn_nsdl_addr_s));
    received_packet_address.addr_ptr = received_address;
    server.set_blocking(false, 50);
    
    //Check incoming socket...
    int n = 0;
    int32_t time = 0;
    
    //Start Timers

//    nsdlTickTimer.start();
    registrationTimer.start();
    #if HEART_BEAT
    heartBeatTimer.start();
    #endif //Heartbeat
    #if NODE_SENSOR_STATION
    temperatureReportTimer.start();
    microphoneSampleTimer.start();
    microphoneReportTimer.start();
    #if NODE_PIR_STATION
    motionReportTimer.start();
    #endif //NODE PIR STATION
    #if NODE_KIOSK_STATION
    kioskReportTimer.start();
    #endif //NODE KIOSK STATION
    #if NODE_DOOR_TRIP_STATION
    doorTripReportTimer.start();
    #endif //NODE TRIP STATION
    #if NODE_HEIGHT_STATION
    heightReportTimer.start();
    #endif //NODE HEIGHT STATION
    #endif //NODE_SENSOR_STATION
    DigitalOut led2(LED2);
    //Watchdog wdt;
//    wdt.kick(WATCHDOG_TIMEOUT_SEC);
    NSP_registration();
    while(true) {
        //Kick the watchdog...
//        wdt.kick();
        //UDP Packet Receive...
        n = server.receiveFrom(from, buffer, sizeof(buffer));
        if (n <= 0) {
            //No Data
        } else { 
            //UDP
            //wait(0.25); //Waiting seems to increase reliability of comms...
            pc.printf("Received %d bytes\r\n", n);
            nsdl_result = sn_nsdl_process_coap((uint8_t*)buffer, n, &received_packet_address);
            pc.printf("Processed COAP Packet: %d\r\n", nsdl_result);
            n = 0;
        }
        
        //NSDL Tick
//        if(nsdlTickTimer.read_ms() >= NSDL_TICK_PERIOD_MS) {
//            sn_nsdl_exec(time);
//            nsdlTickTimer.reset();
//        }
//                 
        //Registration Tick
        if(registrationTimer.read_ms() >= RD_UPDATE_PERIOD_MS) {
            pc.printf("Time to register...\r\n");                                    
            NSP_registration();
            registrationTimer.reset();
        }
        #if HEART_BEAT
        if (heartBeatTimer.read_ms() >= HEARTBEAT_PERIOD_MS){
            //Toggle heartbeat led...
            led2 = !led2;
            heartBeatTimer.reset();
        }
        #endif //HEart Beat
        #if NODE_SENSOR_STATION
        if (temperatureReportTimer.read_ms() >= TEMPERATURE_REPORT_PERIOD_MS){
            //debug("Event: Temperature Report Timer\r\n");
            handle_temperature_report_timer();
            temperatureReportTimer.reset();
        }
        if (microphoneSampleTimer.read_ms() >= SOUND_SAMPLE_PERIOD_MS){
            handle_microphone_sample_timer();
            microphoneSampleTimer.reset();
        }
        if (microphoneReportTimer.read_ms() >= SOUND_REPORT_PERIOD_MS){
            //debug("Event: Sound Report Timer\r\n");
            handle_microphone_report_timer();
            microphoneReportTimer.reset();
        }
        #if NODE_PIR_STATION
        if (motionReportTimer.read_ms() >= MOTION_REPORT_PERIOD_MS){
            //debug("Event: Motion Report Timer\r\n");
            handle_motion_report_timer();
            motionReportTimer.reset();
        }
        #endif //NODE PIR STATION
        #if NODE_KIOSK_STATION
        if (kioskReportTimer.read_ms() >= KIOSK_REPORT_PERIOD_MS){
            //debug("Event: Motion Report Timer\r\n");
            handle_kiosk_report_timer();
            kioskReportTimer.reset();
        }
        #endif //NODE KIOSK STATION
        #if NODE_DOOR_TRIP_STATION
        if (doorTripReportTimer.read_ms() >= DOOR_TRIP_REPORT_PERIOD_MS){
            //debug("Event: Door Trip Report Timer\r\n");
            handle_door_trip_report_timer();
            doorTripReportTimer.reset();
        }
        #endif //NODE TRIP STATION
        #if NODE_HEIGHT_STATION
        if (heightReportTimer.read_ms() >= DOOR_HEIGHT_PERIOD_MS ){
            handle_door_height_sample_timer();
            heightReportTimer.reset();
        }
        #endif //NODE HEIGHT STATION
        #endif //NODE_SENSOR_STATION
    }
    
    
}



/**
 *  \param none
 *  \return int
 */
int main() {
    pc.printf("Initialising Ethernet...\r\n");
    // Initialize Ethernet interface first
    ethernet_init();
    pc.printf("Initialising NSDL...\r\n");
    //Run NSDL...
    nsdl_run();
    //Init Sensors
    init_sensors();
    

    // Run the event loop (never returns)
    main_event_loop();
}
