/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "mbed.h"
#include "WiflyInterface.h"
#include "main.h"
#include "nsdl_support.h"
#include "switch.h"

//Serial pc(USBTX, USBRX);

/* wifly object where:
*     - p9 and p10 are for the serial communication
*     - p25 is for the reset pin
*     - p26 is for the connection status
*     - "mbed" is the ssid of the network
*     - "password" is the password
*     - WPA is the security
*/
WiflyInterface wifly(p9, p10, p25, p26, SSID, AP_KEY, WPA);
//WiflyInterface wifly(PTE16, PTE17, PTD4, PTD2, SSID, AP_KEY, WPA); //KL46Z

// NSP configuration
/* Change this IP address to that of your NanoService Platform installation */
Endpoint nsp;
UDPSocket server;
//extern TCPSocketConnection server;
char endpoint_name[] = {"switch"};
uint8_t ep_type[] = {"mbed_LPC1768"};
uint8_t lifetime_ptr[] = {"86400"};

//static const char* NSP_ADDRESS = "208.111.39.209"; /* demo NSP, web interface at http://nanoservice-demo.mbed.org*/ 
//static const char* NSP_ADDRESS = "10.2.131.119"; /* aseserver NSP, */ 
static const char* NSP_ADDRESS = "192.168.1.10"; /* Trenton BBB NSP */ 
static const int NSP_PORT = 5683;

//Hard Fault Handler (Watchdog)
extern "C" void HardFault_Handler() {
    printf("Hard Fault!\r\n");
    NVIC_SystemReset();
}

// ****************************************************************************
// NSP initialization

static void nsp_connect()
{
    printf("EP Name: %s\r\n", endpoint_name);
    printf("NSP Location: coap://%s:%d\r\n", NSP_ADDRESS, NSP_PORT);
    
    // Bind the port
    //cellular->bind(EP_PORT);
    server.init();
    //server.connect(NSP_ADDRESS, NSP_PORT);
    server.bind(NSP_PORT);
    nsp.set_address(NSP_ADDRESS, NSP_PORT);

    printf("UDP connection to NSP successful.\r\n");  
}

static int create_resources()
{
    sn_nsdl_resource_info_s *resource_ptr = NULL;
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
    
    printf("Creating resources\r\n");

    /* Create resources */
    resource_ptr = (sn_nsdl_resource_info_s*)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
    if(!resource_ptr)
        return 0;
    memset(resource_ptr, 0, sizeof(sn_nsdl_resource_info_s));

    resource_ptr->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
    if(!resource_ptr->resource_parameters_ptr)
    {
        nsdl_free(resource_ptr);
        return 0;
    }
    memset(resource_ptr->resource_parameters_ptr, 0, sizeof(sn_nsdl_resource_parameters_s));

    // Static resources
    nsdl_create_static_resource(resource_ptr, sizeof("detail/name")-1, (uint8_t*) "detail/name", 0, 0,  (uint8_t*) "LPC1768 Switch", sizeof("LPC1768 Switch")-1);

    // Dynamic resources
    create_switch_resource(resource_ptr);

    /* Register with NSP */
    endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)endpoint_name, ep_type, lifetime_ptr);
    if(sn_nsdl_register_endpoint(endpoint_ptr) != 0) {
        printf("NSP registering failed\r\n");
    } else {
        printf("NSP registering OK\r\n");
    }
    nsdl_clean_register_endpoint(&endpoint_ptr);

    nsdl_free(resource_ptr->resource_parameters_ptr);
    nsdl_free(resource_ptr);
    return 1;
}

/* The number of seconds between NSDL Ticks*/
#define NSDL_TICK_PERIOD  1
/* The number of seconds between NSP registration messages */
#define RD_UPDATE_PERIOD  300

void nsdl_event_loop()
{
    //Thread registration_thread(registration_update_thread);
    
    //For timing control
    Timer nsdlTickTimer;
    Timer registrationTimer;
    
    //Re-registration
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
    
    //For recieving NSP messages
    sn_nsdl_addr_s received_packet_address;
    uint8_t received_address[4];
    int8_t nsdl_result = 0;
    char buffer[256];
    Endpoint from;
    memset(&received_packet_address, 0, sizeof(sn_nsdl_addr_s));
    received_packet_address.addr_ptr = received_address;
    server.set_blocking(false, 1500);
    
    //Check incoming socket...
    int n = 0;
    int32_t time = 0;
    nsdlTickTimer.start();
    registrationTimer.start();
    while(true)
    {
        //Wifly UDP Packet Receive...
        n = server.receiveFrom(from, buffer, sizeof(buffer));
        if (n < 0)
        {
            //No Data
            //printf("Socket error\n\r");
        }
        else
        { 
            //UDP
            //wait(0.25); //Waiting seems to increase reliability of comms...
            printf("Received %d bytes\r\n", n);
            nsdl_result = sn_nsdl_process_coap((uint8_t*)buffer, n, &received_packet_address);
            printf("Processed COAP Packet: %d\r\n", nsdl_result);
            n = 0;
        }
        
        //Check if need to send pressure mat update...
        switch_report();
        
        //NSDL Tick
        if(nsdlTickTimer.read() >= NSDL_TICK_PERIOD) {
            sn_nsdl_exec(time);
            nsdlTickTimer.reset();
        }
                 
        //Registration Tick
        if(registrationTimer.read() >= RD_UPDATE_PERIOD) {
            printf("NSP attempt re-register...\r\n");
            endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)endpoint_name, ep_type, lifetime_ptr);
            if(sn_nsdl_register_endpoint(endpoint_ptr) != 0)
                printf("NSP re-registering failed\r\n");
            else
                printf("NSP re-registering OK\r\n");
            nsdl_clean_register_endpoint(&endpoint_ptr);
            registrationTimer.reset();
        }
    }
    
    
}


/**
 *  \param none
 *  \return int
 */
int main() {
    //factory reset Wifly
    wifly.reset();
    printf("\nFactory Reset\r\n");
    //reboot wifly
    bool success = wifly.reboot();
    printf("Reboot: %d\r\n", success); //success = 1 -> successful process
    printf("Initialising Wifly...\r\n");
    wifly.init(); // use DHCP
    while (!wifly.connect()); // join the network
    printf("IP Address is %s\r\n", wifly.getIPAddress());
        
    // Bind the socket and configure NSP settings
    nsp_connect();
 
    // Initalize NanoService library
    nsdl_init();

    // Create resources & register with NSP
    create_resources();
    // Run the NSDL event loop (never returns)
    nsdl_event_loop();
}
