// switch resource implementation

#include "mbed.h"
#include "nsdl_support.h"
#include "switch.h"

#define SWITCH_RES_ID     "sen/switch"

#define MINIMUM_POLL_PERIOD       1  //Seconds
#define MINIMUM_REPORT_PERIOD    10  //Seconds
#define MINIMUM_DEBOUNCE_PERIOD 200  //ms

DigitalOut led1(LED1);
InterruptIn switch_in(p21);
Timer debounceTimer;
Timer reportTimer;
Timer pollTimer;
/* stored data for observable resource */
static uint8_t obs_number = 0;
static uint8_t *obs_token_ptr = NULL;
static uint8_t obs_token_len = 0;
static uint8_t current_switch = 0;
static uint8_t last_reported_switch = 99;
static char switch_val[2];


/* Interrupt handler for switch mat pin */
/* Handles Interrupt, sets state for main polling thread to send update message. */
void switch_interrupt(){
    if(debounceTimer.read_ms() > MINIMUM_DEBOUNCE_PERIOD) {
        led1 = switch_in;
        current_switch = switch_in;
        debounceTimer.reset();
    }
}

//This is to be called from main program loop... it only sends report if the switch mat has changed.
void switch_report() {
    //Poll switch anyways...
    if(pollTimer.read() > MINIMUM_POLL_PERIOD) {
        led1 = switch_in;
        current_switch = switch_in;
        pollTimer.reset();
    }
    if(reportTimer.read() > MINIMUM_REPORT_PERIOD) {
        //We haven't reported for minimum period, so take a reading and report.
        //led1 = switch_in;
        //current_switch = switch_in;
        last_reported_switch = current_switch + 10; //ensure different values...
    }
    if(last_reported_switch != current_switch) {
        if(obs_number != 0){// && obs_token_ptr != NULL){
            obs_number++;
            snprintf(switch_val,2,"%d" ,current_switch);
            if(sn_nsdl_send_observation_notification(obs_token_ptr, obs_token_len, (uint8_t*)switch_val, 1, &obs_number, 1, COAP_MSG_TYPE_NON_CONFIRMABLE, 0) == 0) {
                printf("switch Observation Sending Failed\r\n");
            } else {
                last_reported_switch = current_switch;
                printf("switch Observation Sent\r\n");
            }
            reportTimer.reset();
        }
    }
}

/* Only GET method allowed */
/* Observable resource */
static uint8_t switch_resource_cb(sn_coap_hdr_s *received_coap_ptr, sn_nsdl_addr_s *address, sn_proto_info_s * proto)
{
    uint8_t switch_reading = switch_in;
    snprintf(switch_val,2,"%d" ,switch_reading);
    sn_coap_hdr_s *coap_res_ptr = 0;

    printf("switch mat callback\r\n");
    coap_res_ptr = sn_coap_build_response(received_coap_ptr, COAP_MSG_CODE_RESPONSE_CONTENT);

    coap_res_ptr->payload_len = 1;
    coap_res_ptr->payload_ptr = (uint8_t*)switch_val;

    if(received_coap_ptr->token_ptr)
    {
        printf("   Token included\r\n");
        if(obs_token_ptr)
        {
            free(obs_token_ptr);
            obs_token_ptr = 0;
        }
        obs_token_ptr = (uint8_t*)malloc(received_coap_ptr->token_len);
        if(obs_token_ptr)
        {
            memcpy(obs_token_ptr, received_coap_ptr->token_ptr, received_coap_ptr->token_len);
            obs_token_len = received_coap_ptr->token_len;
        }
    }

    if(received_coap_ptr->options_list_ptr->observe)
    {
        coap_res_ptr->options_list_ptr = (sn_coap_options_list_s*)malloc(sizeof(sn_coap_options_list_s));
        memset(coap_res_ptr->options_list_ptr, 0, sizeof(sn_coap_options_list_s));
        coap_res_ptr->options_list_ptr->observe_ptr = &obs_number;
        coap_res_ptr->options_list_ptr->observe_len = 1;
        obs_number++;
    }
    printf("   Send observation %d... \r\n", obs_number);
    sn_nsdl_send_coap_message(address, coap_res_ptr);

    coap_res_ptr->options_list_ptr->observe_ptr = 0;
    sn_coap_parser_release_allocated_coap_msg_mem(coap_res_ptr);
    return 0;
}

int create_switch_resource(sn_nsdl_resource_info_s *resource_ptr)
{
    nsdl_create_dynamic_resource(resource_ptr, sizeof(SWITCH_RES_ID)-1, (uint8_t*)SWITCH_RES_ID, 0, 0, 1, &switch_resource_cb, SN_GRS_GET_ALLOWED);
    obs_number++;
    
    //Attach interrupt handler and start debounce...
    debounceTimer.start();
    switch_in.rise(&switch_interrupt);
    switch_in.fall(&switch_interrupt);
    
    reportTimer.start();
    pollTimer.start();
    
    return 0;
}