// Pressure Mat resource implementation

#include "mbed.h"
#include "nsdl_support.h"
#include "pressure_mat.h"

#define PRESSURE_MAT_RES_ID     "sen/pressure_mat"

#define MINIMUM_POLL_PERIOD       1  //Seconds
#define MINIMUM_REPORT_PERIOD    10  //Seconds

//InterruptIn pressure_mat_in(PTD3);
InterruptIn pressure_mat_in(PTA13);
DigitalOut led1(LED1);
Timer debounce;
Timer reportTimer;
Timer pollTimer;
/* stored data for observable resource */
static uint8_t obs_number = 0;
static uint8_t *obs_token_ptr = NULL;
static uint8_t obs_token_len = 0;
static uint8_t current_pressure_mat = 0;
static uint8_t last_reported_pressure_mat = 99;
static char pressure_mat_val[2];


/* Interrupt handler for pressure mat pin */
/* Handles Interrupt, sets state for main polling thread to send update message. */
void pressure_mat_interrupt(){
    if(debounce.read_ms() > 200) {
        current_pressure_mat = !pressure_mat_in;
        led1 = !current_pressure_mat;
        debounce.reset();
    }
}

void pressure_mat_resetTokens() {
    obs_number = 0;
    *obs_token_ptr = NULL;
    obs_token_len = 0;    
}

//This is to be called from main program loop... it only sends report if the pressure mat has changed.
void pressure_mat_report() {
    //Poll mat anyways...
    if(pollTimer.read() > MINIMUM_POLL_PERIOD) {
        current_pressure_mat = !pressure_mat_in;
        led1 = !current_pressure_mat;
        pollTimer.reset();
    }
    if(reportTimer.read() > MINIMUM_REPORT_PERIOD) {
        //We haven't reported for minimum period, so take a reading and report.
        last_reported_pressure_mat = current_pressure_mat + 10; //ensure different values to force report...
        reportTimer.reset();
    }
    if(last_reported_pressure_mat != current_pressure_mat) {
        if(obs_number != 0){// && obs_token_ptr != NULL){
            obs_number++;
            snprintf(pressure_mat_val,2,"%d" ,current_pressure_mat);
            if(sn_nsdl_send_observation_notification(obs_token_ptr, obs_token_len, (uint8_t*)pressure_mat_val, 1, &obs_number, 1, COAP_MSG_TYPE_NON_CONFIRMABLE, 0) == 0)
                printf("Pressure Observation Sending Failed\r\n");
            else
                printf("Pressure Observation Sent\r\n");
                
            //Set last reported to current regardless...
            last_reported_pressure_mat = current_pressure_mat;
        }
    }
}

/* Only GET method allowed */
/* Observable resource */
static uint8_t pressure_mat_resource_cb(sn_coap_hdr_s *received_coap_ptr, sn_nsdl_addr_s *address, sn_proto_info_s * proto)
{
    uint8_t pressure_mat_reading = !pressure_mat_in;
    led1 = !pressure_mat_reading;
    snprintf(pressure_mat_val,2,"%d" ,pressure_mat_reading);
    sn_coap_hdr_s *coap_res_ptr = 0;

    printf("pressure mat callback\r\n");
    coap_res_ptr = sn_coap_build_response(received_coap_ptr, COAP_MSG_CODE_RESPONSE_CONTENT);

    coap_res_ptr->payload_len = 1;
    coap_res_ptr->payload_ptr = (uint8_t*)pressure_mat_val;

    if(received_coap_ptr->token_ptr)
    {
        printf("   Token included\r\n");
        if(obs_token_ptr)
        {
            free(obs_token_ptr);
            obs_token_ptr = 0;
        }
        obs_token_ptr = (uint8_t*)malloc(received_coap_ptr->token_len);
        if(obs_token_ptr)
        {
            memcpy(obs_token_ptr, received_coap_ptr->token_ptr, received_coap_ptr->token_len);
            obs_token_len = received_coap_ptr->token_len;
        }
    }

    if(received_coap_ptr->options_list_ptr->observe)
    {
        coap_res_ptr->options_list_ptr = (sn_coap_options_list_s*)malloc(sizeof(sn_coap_options_list_s));
        memset(coap_res_ptr->options_list_ptr, 0, sizeof(sn_coap_options_list_s));
        coap_res_ptr->options_list_ptr->observe_ptr = &obs_number;
        coap_res_ptr->options_list_ptr->observe_len = 1;
        obs_number++;
    }
    printf("   Send observation %d... \r\n", obs_number);
    sn_nsdl_send_coap_message(address, coap_res_ptr);

    coap_res_ptr->options_list_ptr->observe_ptr = 0;
    sn_coap_parser_release_allocated_coap_msg_mem(coap_res_ptr);
    return 0;
}

int create_pressure_mat_resource(sn_nsdl_resource_info_s *resource_ptr)
{
    nsdl_create_dynamic_resource(resource_ptr, sizeof(PRESSURE_MAT_RES_ID)-1, (uint8_t*)PRESSURE_MAT_RES_ID, 0, 0, 1, &pressure_mat_resource_cb, SN_GRS_GET_ALLOWED);
    obs_number++;
    pressure_mat_in.mode(PullUp);
    //Attach interrupt handler and start debounce...
    debounce.start();
    pressure_mat_in.rise(&pressure_mat_interrupt);
    pressure_mat_in.fall(&pressure_mat_interrupt);
    
    reportTimer.start();
    pollTimer.start();
    
    return 0;
}